package org.mule.connectivity.restconnect.api;

import org.mule.connectivity.restconnect.internal.model.ConnectorCategory;
import org.mule.connectivity.restconnect.internal.model.MavenDependency;
import org.mule.connectivity.restconnect.internal.model.api.RestConnectAPIModel;
import org.mule.connectivity.restconnect.internal.model.api.RestConnectModelBuilder;
import org.mule.connectivity.restconnect.internal.modelGeneration.ModelGenerator;
import org.mule.connectivity.restconnect.internal.templateEngine.builder.TemplateEngineBuilder;

import java.io.File;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.connectivity.restconnect.internal.templateEngine.builder.DevKitTemplateEngineBuilder.createDevKitConnector;
import static org.mule.connectivity.restconnect.internal.templateEngine.builder.SmartConnectorTemplateEngineBuilder.createSmartConnector;

public class RestConnect {
    private File spec;
    private SpecFormat format;
    private Parser parser;
    private ConnectorType connectorType;
    private String apiName;
    private Path outputDir;
    private boolean packageConnector = true;
    private String category;
    private boolean generateProjectFiles = true;
    private String projectDescription;
    private List<MavenDependency> dependencies = new ArrayList<>();
    private String rootDir;
    private ConnectorCategory connectorCategory;
    private String customPackage;
    private String groupId;
    private String artifactId;
    private String version;

    private boolean useSecureFileLoader;
    private String proxyUrl;
    private String proxyReferer;

    private RestConnect(){
        System.setProperty("javax.xml.accessExternalDTD", "file");
        System.setProperty("javax.xml.accessExternalSchema", "file");
    }

    public static RestConnect getInstance(){
        return new RestConnect();
    }

    public RestConnect withApiName(String apiName) {
        this.apiName = apiName;
        return this;
    }

    public RestConnect withOutputDir(Path outputDir) {
        this.outputDir = outputDir;
        return this;
    }

    public RestConnect createConnectorFromSpec(File spec, SpecFormat format, Parser parser, ConnectorType connectorType){
        this.spec = spec;
        this.format = format;
        this.parser = parser;
        this.connectorType = connectorType;

        return this;
    }

    public RestConnect withSecureFileLoader(boolean useSecureFileLoader){
        this.useSecureFileLoader = useSecureFileLoader;
        return this;
    }

    public RestConnect withSecureProxy(String proxyUrl, String proxyReferer){
        this.proxyUrl = proxyUrl;
        this.proxyReferer = proxyReferer;
        return this;
    }

    public void run() throws Exception {
        RestConnectAPIModel model = getModel();

        run(model, getTemplateEngine(connectorType), outputDir, generateProjectFiles, packageConnector);
    }

    public static void run(RestConnectAPIModel model, TemplateEngineBuilder templateEngine, Path outputDir, boolean generateProjectFiles, boolean packageConnector) throws Exception {
        TemplateEngineBuilder templateEngineBuilder = templateEngine.fromModel(model);

        if(outputDir != null){
            templateEngineBuilder.inOutputDir(outputDir);
        }

        templateEngineBuilder.withGenerateProjectFiles(generateProjectFiles);

        templateEngineBuilder.execute(packageConnector);
    }

    private ModelGenerator getModelGenerator(Parser parser, SpecFormat format){
        return ModelGenerator.get(format, parser);
    }

    public static TemplateEngineBuilder getTemplateEngine(ConnectorType connectorType){
        if(connectorType.equals(ConnectorType.DevKitConnector)){
            return createDevKitConnector();
        }
        else{
            return createSmartConnector();
        }
    }

    public RestConnect withPackageConnector(boolean packageConnector) {
        this.packageConnector = packageConnector;
        return this;
    }

    public RestConnect withCategory(String category) {
        this.category = category;
        return this;
    }

    public RestConnect withGenerateProjectFiles(boolean generateProjectFiles) {
        this.generateProjectFiles = generateProjectFiles;
        return this;
    }

    public RestConnect withprojectDescription(String projectDescription) {
        this.projectDescription = projectDescription;
        return this;
    }

    public RestConnect withDependency(MavenDependency dependency) {
        this.dependencies.add(dependency);
        return this;
    }

    public RestConnect withRootDir(String rootDir) {
        this.rootDir = rootDir;
        return this;
    }

    public RestConnect withCategory(ConnectorCategory connectorCategory) {
        this.connectorCategory = connectorCategory;
        return this;
    }

    public RestConnect withCustomPackage(String customPackage) {
        this.customPackage = customPackage;
        return this;
    }

    public RestConnectAPIModel getModel() throws Exception {
        ModelGenerator modelGenerator = ModelGenerator.get(format, parser);

        if(isNotBlank(proxyUrl)){
            modelGenerator.withSafeProxy(proxyUrl, proxyReferer);
        }
        modelGenerator.withSafeFileLoader(useSecureFileLoader);

        RestConnectModelBuilder modelBuilder;
        if(isNotBlank(rootDir)){
            modelBuilder = modelGenerator.generateAPIModel(spec, rootDir);
        }
        else {
            modelBuilder = modelGenerator.generateAPIModel(spec);
        }

        if(isNotBlank(apiName)){
            modelBuilder.withApiName(apiName);
        }

        if(isNotBlank(category)){
            modelBuilder.withCategory(category);
        }

        if(isNotBlank(projectDescription)){
            modelBuilder.withProjectDescription(projectDescription);
        }

        if(isNotBlank(customPackage)){
            modelBuilder.withCustomPackage(customPackage);
        }

        if(isNotBlank(groupId)){
            modelBuilder.withGroupId(groupId);
        }

        if(isNotBlank(artifactId)){
            modelBuilder.withArtifactId(artifactId);
        }

        if(isNotBlank(version)){
            modelBuilder.withVersion(version);
        }

        if(connectorCategory != null){
            modelBuilder.withCategory(connectorCategory);
        }

        for(MavenDependency dependency : dependencies){
            modelBuilder.withDependency(dependency);
        }

        return modelBuilder.build();
    }

    public RestConnect withGroupId(String groupId) {
        this.groupId = groupId;
        return this;
    }

    public RestConnect withArtifactId(String artifactId) {
        this.artifactId = artifactId;
        return this;
    }

    public RestConnect withVersion(String version) {
        this.version = version;
        return this;
    }
}
