package org.mule.connectivity.restconnect.internal.templateEngine.decorator.security.smartconnector;

import org.apache.commons.lang3.StringUtils;
import org.mule.connectivity.restconnect.internal.model.MavenDependency;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.type.SmartConnectorTypeDefinitionDecorator;

import java.util.LinkedList;
import java.util.List;

public class SmartConnectorMergedSchemeDecorator {

    private final List<SmartConnectorSecuritySchemeDecorator> securitySchemes;
    private final List<SmartConnectorXMLConfig> xmlConfigs;
    private final List<SmartConnectorXMLSchema> xmlSchemas;
    private final List<SmartConnectorXMLSchemaLocation> xmlSchemaLocations;
    private final List<MavenDependency> pomDependencies;
    private final String modulePrefix;

    public SmartConnectorMergedSchemeDecorator(List<SmartConnectorSecuritySchemeDecorator> securitySchemes, String modulePrefix) {
        this.securitySchemes = securitySchemes;
        this.xmlConfigs = buildXmlConfigs();
        this.xmlSchemas = buildXmlSchemas();
        this.xmlSchemaLocations = buildXmlSchemaLocations();
        pomDependencies = buildPomDependencies();
        this.modulePrefix = modulePrefix;
    }

    private List<MavenDependency> buildPomDependencies() {
        List<MavenDependency> dependencies = new LinkedList<>();
        for(SmartConnectorSecuritySchemeDecorator scheme : securitySchemes){
            for(MavenDependency dependency : scheme.getPomDependencies()){
                if(dependencies.stream().noneMatch(x -> x.getArtifactId().equals(dependency.getGroupId()) && x.getGroupId().equals(dependency.getGroupId()))){
                    dependencies.add(dependency);
                }
            }
        }
        return dependencies;
    }

    protected List<SmartConnectorXMLSchemaLocation> buildXmlSchemaLocations() {
        List<SmartConnectorXMLSchemaLocation> locations = new LinkedList<>();

        for(SmartConnectorSecuritySchemeDecorator scheme : securitySchemes){
            for(SmartConnectorXMLSchemaLocation location : scheme.xmlSchemaLocations){
                if(!locations.contains(location)){
                    locations.add(location);
                }
            }
        }

        return locations;
    }

    protected List<SmartConnectorXMLSchema> buildXmlSchemas() {
        List<SmartConnectorXMLSchema> xmlSchemas = new LinkedList<>();

        for(SmartConnectorSecuritySchemeDecorator scheme : securitySchemes){
            for(SmartConnectorXMLSchema xmlSchema : scheme.xmlSchemas){
                if(!xmlSchemas.contains(xmlSchema)){
                    xmlSchemas.add(xmlSchema);
                }
            }
        }

        return xmlSchemas;
    }

    protected List<SmartConnectorXMLConfig> buildXmlConfigs() {
        List<SmartConnectorXMLConfig> xmlConfigs = new LinkedList<>();

        for(SmartConnectorSecuritySchemeDecorator scheme : securitySchemes){
            for(SmartConnectorXMLConfig xmlConfig : scheme.xmlConfigs){
                if(!xmlConfigs.contains(xmlConfig)){
                    xmlConfigs.add(xmlConfig);
                }
            }
        }

        return xmlConfigs;
    }

    public String getXmlHttpConfigString(boolean oAuthSchemaExists){

        StringBuilder builder = new StringBuilder();

        boolean setConnection = true;
        for(SmartConnectorSecuritySchemeDecorator scheme : securitySchemes){
            String xmlHttpConfigString = scheme.getXmlHttpConfigString(modulePrefix, setConnection, !oAuthSchemaExists);

            if(StringUtils.isNotEmpty(xmlHttpConfigString)){
                builder.append(xmlHttpConfigString).append(System.lineSeparator());
                setConnection = false;
            }
        }

        return builder.toString();
    }

    public String getXmlConfigsString(){
        return SmartConnectorSecuritySchemeDecoratorHelper.getXmlConfigsString(xmlConfigs);
    }

    public String getXmlSchemasString(){
        return SmartConnectorSecuritySchemeDecoratorHelper.getXmlSchemasString(xmlSchemas);
    }

    public String getXmlSchemaLocationsString(){
        return SmartConnectorSecuritySchemeDecoratorHelper.getXmlSchemaLocationsString(xmlSchemaLocations);
    }

    public List<SmartConnectorTypeDefinitionDecorator> getConfigProperties(){
        List<SmartConnectorTypeDefinitionDecorator> allConfigProperties = new LinkedList<>();

        for(SmartConnectorSecuritySchemeDecorator scheme : securitySchemes){
            allConfigProperties.addAll(scheme.getConfigProperties());
        }

        return allConfigProperties;
    }

    public List<MavenDependency> getPomDependencies() {
        return pomDependencies;
    }
}
