package org.mule.connectivity.restconnect.internal.model.security;

import org.mule.connectivity.restconnect.internal.model.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.model.type.TypeDefinitionBuilder;
import org.mule.connectivity.restconnect.internal.model.typesource.PrimitiveTypeSource;

import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import static java.util.function.Predicate.isEqual;
import static java.util.stream.Collectors.toList;
import static javax.ws.rs.core.MediaType.TEXT_PLAIN_TYPE;
import static org.mule.connectivity.restconnect.internal.model.parameter.ParameterType.SECURITY;

public abstract class APISecurityScheme {

    public static final String BASIC = "Basic Authentication";
    public static final String OAUTH1 = "OAuth 1.0"; // Not supported
    public static final String OAUTH2 = "OAuth 2.0";
    public static final String PASS_THROUGH = "Pass Through";
    public static final String DIGEST_AUTHENTICATION = "Digest Authentication";
    public static final String CUSTOM_AUTHENTICATION = "Custom Authentication";
    public static final String JWT = "Jwt";
    public static final String CUSTOM_JWT = "x-Jwt";
    public static final String UNSECURED = "";

    protected List<Parameter> configParameters = new LinkedList<>();
    protected List<Parameter> headers = new LinkedList<>();
    protected List<Parameter> queryParameters = new LinkedList<>();

    public APISecurityScheme() {
        initializeConfigParameters();
    }

    public abstract String getSchemeName();

    public abstract void initializeConfigParameters();

    public List<Parameter> getConfigParameters(){
        return configParameters;
    }

    public List<Parameter> getQueryParameters(){
        return queryParameters;
    }

    public List<Parameter> getHeaders(){
        return headers;
    }

    public boolean hasQueryParameters(){
        return !queryParameters.isEmpty();
    }

    public boolean hasHeaders(){
        return !headers.isEmpty();
    }

    public boolean hasConfigParameters(){
        return !configParameters.isEmpty();
    }

    public boolean equals(APISecurityScheme scheme){
        return
                scheme.getClass().equals(this.getClass())
                && scheme.getSchemeName().equals(this.getSchemeName())
                && equalProperties(scheme);
    }

    //This method should check if the properties defined for the security schemes are equal
    protected abstract boolean equalProperties(APISecurityScheme scheme);

    public static Parameter getSecuritySchemeParameter(String name,
                                                       String displayName,
                                                       PrimitiveTypeSource.PrimitiveType type,
                                                       String description,
                                                       String defaultValue,
                                                       String example,
                                                       boolean required,
                                                       boolean isPassword,
                                                       String... enumValues) {
        return getSecuritySchemeParameter(name, displayName, type, description, defaultValue, example, required, isPassword, false, enumValues);
    }

    public static Parameter getSecuritySchemeParameter(String name,
                                                       String displayName,
                                                       PrimitiveTypeSource.PrimitiveType type,
                                                       String description,
                                                       String defaultValue,
                                                       String example,
                                                       boolean required,
                                                       boolean isPassword,
                                                       boolean generated,
                                                       String... enumValues) {
        return new Parameter(name,
                SECURITY,
                new TypeDefinitionBuilder(TEXT_PLAIN_TYPE,
                        new PrimitiveTypeSource(type),
                        required,
                        false,
                        false)
                        .withAnnotatedDisplayName(Optional.ofNullable(displayName).orElse(name))
                        .withDescription(description)
                        .withDefaultValue(defaultValue)
                        .withExample(example)
                        .withEnumValues(Optional.ofNullable(enumValues)
                                .filter(isEqual(new String[]{}).negate())
                                .map(Stream::of)
                                .map(stream -> stream.collect(toList()))
                                .orElse(null))
                        .build(),
                isPassword,
                generated);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof APISecurityScheme)) return false;

        APISecurityScheme that = (APISecurityScheme) o;

        if (configParameters != null ? !configParameters.equals(that.configParameters) : that.configParameters != null)
            return false;
        if (headers != null ? !headers.equals(that.headers) : that.headers != null) return false;
        return queryParameters != null ? queryParameters.equals(that.queryParameters) : that.queryParameters == null;
    }

    @Override
    public int hashCode() {
        int result = configParameters != null ? configParameters.hashCode() : 0;
        result = 31 * result + (headers != null ? headers.hashCode() : 0);
        result = 31 * result + (queryParameters != null ? queryParameters.hashCode() : 0);
        return result;
    }
}
