package org.mule.connectivity.restconnect.internal.modelGeneration.amf;

import amf.apicontract.client.platform.model.domain.EndPoint;
import amf.apicontract.client.platform.model.domain.api.WebApi;
import org.apache.commons.lang3.StringUtils;

import org.mule.connectivity.restconnect.exception.GenerationException;
import org.mule.connectivity.restconnect.internal.model.HTTPMethod;
import org.mule.connectivity.restconnect.internal.model.operation.Operation;
import org.mule.connectivity.restconnect.internal.model.operation.OperationBuilder;
import org.mule.connectivity.restconnect.internal.model.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.model.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.modelGeneration.JsonSchemaPool;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import static org.mule.connectivity.restconnect.internal.model.parameter.ParameterType.HEADER;
import static org.mule.connectivity.restconnect.internal.model.parameter.ParameterType.QUERY;
import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.AMFTypeDefinitionFactory.buildInputMetadata;
import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.AMFTypeDefinitionFactory.buildOutputMetadata;
import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.security.AMFSecuritySchemeFactory.getSchemesForOperation;
import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.util.AMFParserUtil.*;

public class AMFOperationGenerator {

    public static Operation generateOperation(WebApi api, EndPoint endPoint, amf.apicontract.client.platform.model.domain.Operation operation, JsonSchemaPool jsonSchemaPool) throws GenerationException {

        String friendlyName = getMethodDisplayName(operation);
        String annotatedName = getAnnotatedOperationName(operation);

        return new OperationBuilder()
                .withFriendlyName(friendlyName)
                .withAnnotatedDisplayName(StringUtils.isNotBlank(annotatedName) ? annotatedName : friendlyName)
                .withCanonicalName(getCanonicalOperationName(endPoint, operation, friendlyName))
                .withDescription(buildOperationDescription(endPoint, operation))
                .withHttpMethod(HTTPMethod.fromString(operation.method().value()))
                .withUri(endPoint.path().value())
                .withUriParams(buildUriParams(endPoint, operation, jsonSchemaPool))
                .withQueryParams(buildQueryParams(operation, jsonSchemaPool))
                .withHeaders(buildHeaders(operation, jsonSchemaPool))
                .withInputMetadata(buildInputMetadata(operation, jsonSchemaPool))
                .withOutputMetadata(buildOutputMetadata(operation, jsonSchemaPool))
                .withSecuritySchemes(getSchemesForOperation(api, endPoint, operation, jsonSchemaPool))
                .withUserSelectedSecuritySchemes(isUserSelectedSecuritySchemes(endPoint, operation, api))
                .build();
    }

    private static String buildOperationDescription(EndPoint endPoint, amf.apicontract.client.platform.model.domain.Operation operation) {
        return operation.description().nonEmpty() ? operation.description().value() : endPoint.description().value();
    }

    private static List<Parameter> buildUriParams(EndPoint endPoint, amf.apicontract.client.platform.model.domain.Operation operation, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<amf.apicontract.client.platform.model.domain.Parameter> uriParameters = new LinkedList<>(endPoint.parameters());
        if(operation.request() != null){
            uriParameters.addAll(operation.request().uriParameters());
        }

        return getParameterList(uriParameters, ParameterType.URI, jsonSchemaPool);
    }

    private static List<Parameter> buildQueryParams(amf.apicontract.client.platform.model.domain.Operation operation, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        if(operation.request() != null){
            return getParameterList(operation.request().queryParameters(), QUERY, jsonSchemaPool);
        }
        return new ArrayList<>();
    }

    private static List<Parameter> buildHeaders(amf.apicontract.client.platform.model.domain.Operation operation, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        if(operation.request() != null){
            return getParameterList(operation.request().headers(), HEADER, jsonSchemaPool);
        }
        return new ArrayList<>();
    }

}
