package org.mule.connectivity.restconnect.internal.templateEngine.decorator.operation;

import com.google.common.collect.ImmutableList;
import org.mule.connectivity.restconnect.internal.model.operation.Operation;
import org.mule.connectivity.restconnect.internal.model.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.model.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.model.security.APISecurityScheme;
import org.mule.connectivity.restconnect.internal.model.type.TypeDefinitionBuilder;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.model.SmartConnectorModelDecorator;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.security.smartconnector.SmartConnectorSecuritySchemeDecorator;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.type.SmartConnectorTypeDefinitionDecorator;
import org.mule.connectivity.restconnect.internal.util.DataWeaveUtils;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Collectors;

import static org.mule.connectivity.restconnect.internal.modelGeneration.util.ParserUtils.getXmlName;
import static org.mule.connectivity.restconnect.internal.modelGeneration.util.ParserUtils.isReservedMuleKeyword;
import static org.mule.connectivity.restconnect.internal.templateEngine.decorator.model.SmartConnectorSecuritySchemeHelper.getPrioritarySecurityScheme;
import static org.mule.connectivity.restconnect.internal.templateEngine.decorator.security.smartconnector.SmartConnectorSecuritySchemeDecoratorFactory.getSecuritySchemeDecorator;


public class SmartConnectorOperationDecorator extends OperationDecorator {

    private final SmartConnectorTypeDefinitionDecorator outputMetadata;
    private final SmartConnectorTypeDefinitionDecorator inputMetadata;
    private final List<SmartConnectorTypeDefinitionDecorator> parameters;
    private final SmartConnectorSecuritySchemeDecorator prioritarySecuritySchemeDecorator;
    private final List<SmartConnectorSecuritySchemeDecorator> allSecuritySchemesDecorators;
    private String allSecuritySchemesFilename;

    public SmartConnectorOperationDecorator(Operation operation, HashMap<APISecurityScheme, String> securitySchemesNames, HashMap<String, SmartConnectorTypeDefinitionDecorator> allTypeDecorators, SmartConnectorModelDecorator smartConnectorModelDecorator) {
        super(operation, smartConnectorModelDecorator);
        this.inputMetadata = buildInputMetadata();
        this.outputMetadata = buildOutputMetadata();

        this.parameters = buildDecoratedParameters(this.getParameters(), allTypeDecorators);

        if(operation.isUserSelectedSecuritySchemes() && operation.getSecuritySchemes().size() > 1){
            this.allSecuritySchemesDecorators = buildDecoratedSecuritySchemes(operation, securitySchemesNames);
            this.prioritarySecuritySchemeDecorator = null;
        }
        else{
            this.prioritarySecuritySchemeDecorator = buildDecoratedPrioritarySecurityScheme(operation, securitySchemesNames);
            this.allSecuritySchemesDecorators = null;
        }
    }

    public static List<SmartConnectorTypeDefinitionDecorator> buildDecoratedParameters(List<Parameter> parameters,
                                                                                       HashMap<String, SmartConnectorTypeDefinitionDecorator> allTypeDecorators) {
        return buildDecoratedParameters(parameters, "", allTypeDecorators);
    }

    public static List<SmartConnectorTypeDefinitionDecorator> buildDecoratedParameters(List<Parameter> parameters, String namePrefix, HashMap<String, SmartConnectorTypeDefinitionDecorator> allTypeDecorators) {
        ImmutableList.Builder<SmartConnectorTypeDefinitionDecorator> builder = ImmutableList.builder();

        for (Parameter param : parameters) {
            SmartConnectorTypeDefinitionDecorator smartConnectorTypeDefinitionDecorator = buildDecoratedParameter(param, namePrefix);
            builder.add(smartConnectorTypeDefinitionDecorator);

            if(smartConnectorTypeDefinitionDecorator.requiresCatalog()){
                String typeName = smartConnectorTypeDefinitionDecorator.getType();

                if(!allTypeDecorators.containsKey(typeName)){
                    allTypeDecorators.put(typeName, smartConnectorTypeDefinitionDecorator);
                }
            }
        }

        return builder.build();
    }

    public static  SmartConnectorTypeDefinitionDecorator buildDecoratedParameter(Parameter param, String namePrefix){
        String prefix = (!namePrefix.isEmpty() ? namePrefix + "-" : "");
        String internalName = prefix + param.getInternalName();

        if(isReservedMuleKeyword(param.getInternalName())){
            internalName = "sc-" + internalName;
        }

        return new SmartConnectorTypeDefinitionDecorator(prefix + param.getExternalName(),
                                                         internalName,
                                                         param.getTypeDefinition(),
                                                         param.isPassword());
    }

    protected static SmartConnectorSecuritySchemeDecorator buildDecoratedPrioritarySecurityScheme(Operation operation, HashMap<APISecurityScheme, String> securitySchemesNames) {
        return getSecuritySchemeDecorator(getPrioritarySecurityScheme(operation.getSecuritySchemes()), securitySchemesNames);
    }

    protected static List<SmartConnectorSecuritySchemeDecorator> buildDecoratedSecuritySchemes(Operation operation, HashMap<APISecurityScheme, String> securitySchemesNames) {
        return operation.getSecuritySchemes()
                .stream()
                .map(x -> getSecuritySchemeDecorator(x, securitySchemesNames))
                .collect(Collectors.toList());
    }

    private SmartConnectorTypeDefinitionDecorator buildOutputMetadata() {
        if(getOutputMetadata() == null)
            return null;

        SmartConnectorTypeDefinitionDecorator outputMetadata = new SmartConnectorTypeDefinitionDecorator(getOutputMetadata());
        outputMetadata.setName(getName() + "-response-data");
        outputMetadata.setType(getName() + "-response-type");
        return outputMetadata;
    }

    private SmartConnectorTypeDefinitionDecorator buildInputMetadata() {
        if (getInputMetadata() == null)
            return null;

        SmartConnectorTypeDefinitionDecorator inputMetadata = new SmartConnectorTypeDefinitionDecorator(getInputMetadata());
        inputMetadata.setName(getName() + "-request-data");
        inputMetadata.setType(getName() + "-request-type");
        return inputMetadata;
    }

    @Override
    public List<Parameter> getHeaders() {
        List<Parameter> baseHeaders = this.operation.getHeaders();

        boolean hasAcceptHeader = false;
        for (Parameter param : baseHeaders){
            if(param.getExternalName().equalsIgnoreCase("accept")){
                hasAcceptHeader = true;
            }
        }

        if(!hasAcceptHeader && this.outputMetadata != null && this.outputMetadata.hasMediaType()){
            List<Parameter> headers = new LinkedList<>();

            Parameter fixedValueHeader = new Parameter("Accept", ParameterType.HEADER, TypeDefinitionBuilder.buildHeaderType());
            fixedValueHeader.setFixedValue(this.outputMetadata.getMediaType());

            headers.add(fixedValueHeader);
            headers.addAll(baseHeaders);

            return headers;
        }

        return baseHeaders;
    }

    public String getName() {
        return getXmlName(getCanonicalName());
    }

    public SmartConnectorTypeDefinitionDecorator getDecoratedInputMetadata() {
        return this.inputMetadata;
    }

    public SmartConnectorTypeDefinitionDecorator getDecoratedOutputMetadata() {
        return this.outputMetadata;
    }

    public List<SmartConnectorTypeDefinitionDecorator> getDecoratedParameters() {
        return this.parameters;
    }

    public String getQueryParametersDW() {
        return DataWeaveUtils.getParametersDW(getQueryParameters(), null);
    }

    public String getUriParametersDW() {
        return DataWeaveUtils.getParametersDW(getUriParameters(), null);
    }

    public String getHeadersDW() {
        return DataWeaveUtils.getParametersDW(getHeaders(), null);
    }

    public boolean hasInputParameters() {
        return getInputMetadata() != null || !getParameters().isEmpty();
    }

    public SmartConnectorSecuritySchemeDecorator getSecurityScheme(){
        return this.prioritarySecuritySchemeDecorator;
    }

    public boolean isUserSelectedSecuritySchemes(){
        //We should only care about the user selecting the security scheme when there are schemes where to choose from.
        if (this.operation.getSecuritySchemes().size() > 1){
            return operation.isUserSelectedSecuritySchemes();
        }

        return false;
    }

    public List<SmartConnectorSecuritySchemeDecorator> getAllSecuritySchemes(){
        return this.allSecuritySchemesDecorators;
    }

    private List<String> getAllSecuritySchemesNames(){
        List<String> names = new LinkedList<>();
        for(SmartConnectorSecuritySchemeDecorator ss : getAllSecuritySchemes()){
            names.add(ss.getFriendlyName());
        }
        return names;
    }

    public String getAllSecuritySchemesNamesEnum(){
       return "\"" + String.join("\",\"", getAllSecuritySchemesNames()) + "\"";
    }

    public void setAllSecuritySchemesFilename(String allSecuritySchemesFilename) {
        this.allSecuritySchemesFilename = allSecuritySchemesFilename;
    }

    public String getAllSecuritySchemesFilename(){
        return this.allSecuritySchemesFilename;
    }
}
