package org.mule.connectivity.restconnect.internal.modelGeneration.amf;

import amf.apicontract.client.platform.AMFBaseUnitClient;
import amf.apicontract.client.platform.AMFConfiguration;
import amf.apicontract.client.platform.OASConfiguration;
import amf.apicontract.client.platform.RAMLConfiguration;
import amf.apicontract.client.platform.model.domain.EndPoint;
import amf.apicontract.client.platform.model.domain.Server;
import amf.apicontract.client.platform.model.domain.api.WebApi;
import amf.core.client.platform.AMFParseResult;
import amf.core.client.platform.model.document.BaseUnit;
import amf.core.client.platform.model.document.Document;
import amf.core.client.platform.resource.FileResourceLoader;
import amf.core.client.platform.resource.HttpResourceLoader;
import amf.core.client.platform.resource.ResourceLoader;
import amf.core.client.platform.validation.AMFValidationReport;
import amf.core.client.platform.validation.AMFValidationResult;
import org.apache.logging.log4j.Logger;

import org.mule.connectivity.restconnect.api.SpecFormat;
import org.mule.connectivity.restconnect.exception.GenerationException;
import org.mule.connectivity.restconnect.exception.InvalidSourceException;
import org.mule.connectivity.restconnect.internal.model.api.RestConnectModelBuilder;
import org.mule.connectivity.restconnect.internal.model.operation.Operation;
import org.mule.connectivity.restconnect.internal.model.uri.BaseUri;
import org.mule.connectivity.restconnect.internal.model.uri.BaseUriBuilder;
import org.mule.connectivity.restconnect.internal.modelGeneration.JsonSchemaPool;
import org.mule.connectivity.restconnect.internal.modelGeneration.ModelGenerator;
import org.mule.connectivity.restconnect.internal.modelGeneration.amf.resourceLoader.AMFExchangeDependencyResourceLoader;
import org.mule.connectivity.restconnect.internal.modelGeneration.amf.resourceLoader.SafeFileResourceLoader;
import org.mule.connectivity.restconnect.internal.modelGeneration.amf.resourceLoader.SafeProxyResourceLoader;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.ExecutionException;

import static java.lang.System.lineSeparator;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.stream.Collectors.joining;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.logging.log4j.LogManager.getLogger;
import static org.mule.connectivity.restconnect.api.SpecFormat.JSON_OAS;
import static org.mule.connectivity.restconnect.api.SpecFormat.JSON_OAS3;
import static org.mule.connectivity.restconnect.api.SpecFormat.RAML;
import static org.mule.connectivity.restconnect.api.SpecFormat.YAML_OAS;
import static org.mule.connectivity.restconnect.api.SpecFormat.YAML_OAS3;
import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.util.AMFParserUtil.isIgnored;
import static org.mule.connectivity.restconnect.internal.modelGeneration.util.OperationMappingUtils.disambiguateRepeatedOperations;

public class AMFModelGenerator extends ModelGenerator {
    private final static Logger logger = getLogger(AMFModelGenerator.class);

    public AMFModelGenerator(SpecFormat specFormat) {
        super(specFormat);
    }

    public RestConnectModelBuilder generateAPIModel(File raml, String rootDir) throws GenerationException, ExecutionException, InterruptedException {
        AMFConfiguration configuration = getAMFConfiguration(specFormat, rootDir, proxyUri, proxyReferer, safeFileLoader);
        WebApi webApi = getWebApi(configuration.baseUnitClient(), raml.toPath());

        Path rootDirPath = new File(rootDir).toPath().toAbsolutePath();
        JsonSchemaPool jsonSchemaPool = new JsonSchemaPool();

        return RestConnectModelBuilder.createModel()
                .withRootDir(rootDirPath)
                .withApiDescription(webApi.description().nonEmpty() ? webApi.description().value() : "")
                .withApiName(webApi.name().value())
                .withBaseUri(buildBaseUri(webApi))
                .withOperations(buildOperationsModel(webApi, jsonSchemaPool));
    }

    private static List<Operation> buildOperationsModel(WebApi webApi, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<Operation> operations = new ArrayList<>();

        for (EndPoint endPoint : webApi.endPoints()) {
            for(amf.apicontract.client.platform.model.domain.Operation operation : endPoint.operations()){
                if(isIgnored(endPoint, operation))
                {
                    logger.warn("Resource ignored: {} {}", operation.method(), endPoint.path());
                }
                else
                {
                    operations.add(AMFOperationGenerator.generateOperation(webApi, endPoint, operation, jsonSchemaPool));
                }
            }
        }

        disambiguateRepeatedOperations(operations);
        return operations;
    }

    private static BaseUri buildBaseUri(WebApi api) {
        String url = "";

        for(Server server : api.servers()){
            if(server.url().nonEmpty()){
                url = server.url().value();
                break;
            }
        }

        return BaseUriBuilder.buildBaseUri(url, api.version().value());
    }


    private static AMFConfiguration getAMFConfiguration(SpecFormat format, String rootDir, String proxyBaseUri, String proxyReferer, boolean safeFileLoader) {
        AMFConfiguration amfConfiguration;

        if(format.equals(RAML)){
            amfConfiguration = RAMLConfiguration.RAML10();
        }
        else if(format.equals(JSON_OAS) || format.equals(YAML_OAS)){
            amfConfiguration = OASConfiguration.OAS20();
        }
        else if(format.equals(YAML_OAS3) || format.equals(JSON_OAS3)){
            amfConfiguration = OASConfiguration.OAS30();
        }
        else {
            throw new IllegalArgumentException("Spec format " + format.getName() + "not supported");
        }

        return amfConfiguration
            .withResourceLoaders(getResourceLoaders(rootDir, proxyBaseUri, proxyReferer, safeFileLoader));
    }

    private static List<ResourceLoader> getResourceLoaders(String rootDir, String proxyBaseUri, String proxyReferer, boolean safeFileLoader) {
        List<ResourceLoader> resourceLoaders = new LinkedList<>();

        if(isNotBlank(proxyBaseUri)){
            resourceLoaders.add(new SafeProxyResourceLoader(proxyBaseUri, proxyReferer));
        }
        else{
            resourceLoaders.add(new HttpResourceLoader());
        }

        if(safeFileLoader){
            resourceLoaders.add(new SafeFileResourceLoader(rootDir));
        }
        else{
            resourceLoaders.add(new FileResourceLoader());
        }

        resourceLoaders.add(new AMFExchangeDependencyResourceLoader(rootDir));

        return resourceLoaders;
    }

    public WebApi getWebApi(AMFBaseUnitClient client, Path path){
        if(isRaml08(path.toAbsolutePath())){
            throw new InvalidSourceException("RAML 0.8 is not supported.");
        }

        return parseWebApi(client, path);
    }

    private void validateDocument(AMFBaseUnitClient client, Document document) {
        AMFValidationReport report;

        try {
             report = client.validate(document).get();
        }
        catch (Exception e){
            throw new InvalidSourceException("Could not validate API.");
        }

        if(!report.conforms()){
            throw new InvalidSourceException("The provided API specification has errors.\n" + report);
        }
    }

    private WebApi parseWebApi(AMFBaseUnitClient client, Path path) {
        AMFParseResult amfParseResult;

        try {
            amfParseResult = client.parse("file://" + path.toAbsolutePath()).get();
        } catch (InterruptedException | ExecutionException e) {
            throw new RuntimeException("An error happened while parsing the API specification", e);
        }

        if(!amfParseResult.conforms()){
            throw new InvalidSourceException("The provided API specification has errors.\n" +
                                                 amfParseResult.results().stream()
                                                     .map(AMFValidationResult::message)
                                                     .collect(joining(lineSeparator())));
        }

        BaseUnit baseUnit = client.transform(amfParseResult.baseUnit()).baseUnit();

        Document document;
        if (baseUnit instanceof Document) {
            document = (Document) baseUnit;
        } else {
            throw new InvalidSourceException("Document type not supported: " + baseUnit.getClass());
        }

        validateDocument(client, document);

        return (WebApi) document.encodes();
    }

    private boolean isRaml08(Path specPath){
        String content = "";

        try {
            content = new String(java.nio.file.Files.readAllBytes(specPath), UTF_8);
        } catch (IOException e) {
            e.printStackTrace();
        }

        return content.startsWith("#%RAML 0.8");
    }

}
