package org.mule.connectivity.restconnect.internal.modelGeneration.amf.resourceLoader;

import static java.lang.String.format;
import static java.util.concurrent.CompletableFuture.runAsync;
import static org.mule.connectivity.restconnect.internal.modelGeneration.util.ResourceLoaderUtils.getHttpURLConnection;

import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.concurrent.CompletableFuture;

import amf.core.client.common.remote.Content;
import amf.core.client.platform.resource.BaseHttpResourceLoader;
import amf.core.internal.remote.UnexpectedStatusCode;
import scala.Option;
import org.codehaus.plexus.util.IOUtil;


public class SafeProxyResourceLoader extends BaseHttpResourceLoader {

  private static final String USER_AGENT = "rest-connect";

  private final String proxyBaseUri;
  private final String referer;

  public SafeProxyResourceLoader(String proxyBaseUri, String referer){
    this.proxyBaseUri = proxyBaseUri.endsWith("/") ? proxyBaseUri.substring(0, proxyBaseUri.length()-1) : proxyBaseUri;
    this.referer = referer;
  }

  @Override
  public CompletableFuture<Content> fetch(String resource) {
    if (resource == null || resource.isEmpty()){
      return fail();
    }

    //Create a URI for the resource, so we can check if its absolute or relative.
    //We don't want to make external requests to relative resources.
    URI uri;
    try {
      //Normalize the URI - we always send the same URL format in the requests.
      uri = new URI(resource).normalize();
    } catch (URISyntaxException e) {
      return fail();
    }
    if(!uri.isAbsolute()){
      return fail();
    }
    final String finalResourceUrl = (uri.getScheme() == null ? "http://" : "") + uri.toString();


    CompletableFuture<Content> f = new CompletableFuture<>();

    runAsync(() -> {
      try{

        URL u = new URL(format("%s/%s", proxyBaseUri, finalResourceUrl));

        HttpURLConnection connection = getHttpURLConnection(u, referer);

        Content content = createContent(connection);
        f.complete(content);
      }
      catch (Exception e){
        f.completeExceptionally(e);
      }

      f.completeExceptionally(new Exception("Could not complete future"));
    });

    return f;
  }

  private Content createContent(final HttpURLConnection connection) throws Exception {
    connection.connect();
    int statusCode = connection.getResponseCode();

    if(statusCode == 200){
      Option<String> option = Option.apply(connection.getHeaderField("Content-Type"));
      return new Content(IOUtil.toString(connection.getInputStream()), connection.getURL().toString(), option);
    }
    else{
      throw new UnexpectedStatusCode(connection.getURL().toString(), statusCode);
    }
  }

  private CompletableFuture<Content> fail() {
    return CompletableFuture.supplyAsync(() -> {
      throw new RuntimeException("Failed to apply.");
    });
  }
}
