package org.mule.connectivity.restconnect.internal.modelGeneration.amf.security;

import org.mule.connectivity.restconnect.exception.GenerationException;
import org.mule.connectivity.restconnect.exception.UnsupportedSecuritySchemeException;
import org.mule.connectivity.restconnect.internal.model.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.model.security.*;
import org.mule.connectivity.restconnect.internal.modelGeneration.JsonSchemaPool;
import org.mule.connectivity.restconnect.internal.modelGeneration.common.security.SecuritySchemeFactory;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Stream;

import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static org.mule.connectivity.restconnect.internal.model.parameter.ParameterType.SECURITY;
import static org.mule.connectivity.restconnect.internal.model.security.OAuth2Scheme.OAUTH2_GRANT_AUTHORIZATION_CODE;
import static org.mule.connectivity.restconnect.internal.model.security.OAuth2Scheme.OAUTH2_GRANT_CLIENT_CREDENTIALS;
import static org.mule.connectivity.restconnect.internal.model.type.TypeDefinitionBuilder.buildSimpleStringType;
import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.security.AMFOauth2FlowsNaming.isAuthorizationCode;
import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.security.AMFOauth2FlowsNaming.isClientCredentials;
import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.util.AMFParserUtil.getAnnotatedRenewTokenExpression;
import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.util.AMFParserUtil.getParameterList;

import amf.apicontract.client.platform.model.domain.EndPoint;
import amf.apicontract.client.platform.model.domain.Operation;
import amf.apicontract.client.platform.model.domain.api.WebApi;
import amf.apicontract.client.platform.model.domain.security.ApiKeySettings;
import amf.apicontract.client.platform.model.domain.security.HttpSettings;
import amf.apicontract.client.platform.model.domain.security.OAuth2Flow;
import amf.apicontract.client.platform.model.domain.security.OAuth2Settings;
import amf.apicontract.client.platform.model.domain.security.ParametrizedSecurityScheme;
import amf.apicontract.client.platform.model.domain.security.SecurityRequirement;
import amf.apicontract.client.platform.model.domain.security.SecurityScheme;
import amf.core.client.platform.model.StrField;
import amf.core.client.platform.model.domain.DomainElement;

public class AMFSecuritySchemeFactory {

    public static List<APISecurityScheme> getSchemesForOperation(WebApi api, EndPoint endPoint, Operation operation, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<SecurityScheme> globalSchemes = api.security().stream().flatMap(x -> getSchemes(x)).collect(toList());

        List<SecurityScheme> endPointSchemes = endPoint.security().stream().flatMap(x -> getSchemes(x)).collect(toList());

        List<SecurityScheme> operationSchemes = operation.security().stream().flatMap(x -> getSchemes(x)).collect(toList());

        return getOperationSecuritySchemes(operationSchemes, endPointSchemes, globalSchemes, jsonSchemaPool);
    }

    @SuppressWarnings("Duplicates")
    private static List<APISecurityScheme> getOperationSecuritySchemes(List<SecurityScheme> operationSchemes, List<SecurityScheme> endPointSchemes, List<SecurityScheme> globalSchemes, JsonSchemaPool jsonSchemaPool) throws GenerationException {

        List<SecurityScheme> securitySchemesForOperation = SecuritySchemeFactory.getSecuritySchemesForOperation(operationSchemes, endPointSchemes, globalSchemes);

        if(!securitySchemesForOperation.isEmpty()){
            return getAPISecuritySchemes(securitySchemesForOperation, jsonSchemaPool);
        }
        else{
            //If there are no security schemes defined for this operation, we create an UnsecuredScheme
            List<APISecurityScheme> returnSchemes = new LinkedList<>();
            returnSchemes.add(new UnsecuredScheme());
            return returnSchemes;
        }
    }

    @SuppressWarnings("Duplicates")
    private static List<APISecurityScheme> getAPISecuritySchemes(List<SecurityScheme> sourceSchemes, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<APISecurityScheme> returnSchemes = new LinkedList<>();

        for (SecurityScheme scheme : sourceSchemes) {
            List<APISecurityScheme> generatedSchemes = createSecuritySchemes(scheme, jsonSchemaPool);

            for(APISecurityScheme generatedScheme : generatedSchemes){
                if(returnSchemes.stream().noneMatch(x -> x.equals(generatedScheme))){
                    returnSchemes.add(generatedScheme);
                }
            }
        }

        // If the API is secured but we don't support any defined schemes, then we must throw a generation exception.
        if(!sourceSchemes.isEmpty() && returnSchemes.isEmpty()){
            throw new UnsupportedSecuritySchemeException("None of the specified security schemes ( " + listSchemes(sourceSchemes) + ") are supported.");
        }

        return returnSchemes;
    }

    private static List<APISecurityScheme> createSecuritySchemes(SecurityScheme scheme, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        ArrayList<APISecurityScheme> apiSecuritySchemes = new ArrayList<>();

        if (scheme == null || !scheme.type().nonEmpty()){
            apiSecuritySchemes.add(new UnsecuredScheme());
            return apiSecuritySchemes;
        }

        String schemeType = scheme.type().value();
        String httpScheme = null;
        if(scheme.settings() instanceof HttpSettings){
            httpScheme = ((HttpSettings)scheme.settings()).scheme().value();
        }

        if(AMFSecuritySchemesNaming.isBasicAuth(schemeType, httpScheme)){
            apiSecuritySchemes.add(new BasicAuthScheme());
        }
        else if(AMFSecuritySchemesNaming.isPassThrough(schemeType)){
            apiSecuritySchemes.add(buildPassThroughSecurityScheme(scheme, jsonSchemaPool));
        }
        else if(AMFSecuritySchemesNaming.isApiKey(schemeType)){
            apiSecuritySchemes.add(buildApiKeySecurityScheme(scheme, jsonSchemaPool));
        }
        else if(AMFSecuritySchemesNaming.isOauth2(schemeType)){
            OAuth2Settings oAuth2Settings = (OAuth2Settings)scheme.settings();
            OAuth2Flow oAuth2Flow =
                oAuth2Settings
                    .flows()
                    .stream()
                    .filter(x -> isAuthorizationCode(oAuth2Settings, x) || isClientCredentials(oAuth2Settings, x))
                    .findFirst().orElse(null);


            if(oAuth2Flow != null){
                if (isAuthorizationCode(oAuth2Settings, oAuth2Flow)) {
                    apiSecuritySchemes.add(buildOAuth2AuthorizationCodeSecurityScheme(scheme, oAuth2Flow));
                } else if (isClientCredentials(oAuth2Settings, oAuth2Flow)) {
                    apiSecuritySchemes.add(buildOAuth2ClientCredentialsSecurityScheme(scheme, oAuth2Flow));
                }
            }
        }
        else if(AMFSecuritySchemesNaming.isDigestAuth(schemeType)){
            apiSecuritySchemes.add(new DigestAuthenticationScheme());
        }
        else if (AMFSecuritySchemesNaming.isJwtAuth(schemeType)) {
            apiSecuritySchemes.add(buildJwtAuthenticationScheme(scheme, jsonSchemaPool));
        }
        else if(AMFSecuritySchemesNaming.isCustom(schemeType, httpScheme)){
            apiSecuritySchemes.add(buildCustomAuthenticationScheme(scheme, jsonSchemaPool));
        }

        return apiSecuritySchemes;
    }

    private static Stream<SecurityScheme> getSchemes(DomainElement domainElement) {
        if (domainElement instanceof SecurityRequirement) {
            SecurityRequirement securityRequirement = (SecurityRequirement) domainElement;
            return securityRequirement.schemes().stream().map(ParametrizedSecurityScheme::scheme);
        }
        return Stream.empty();
    }

    private static PassThroughScheme buildPassThroughSecurityScheme(SecurityScheme securityScheme, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<Parameter> queryParameters = getParameterList(securityScheme.queryParameters(), SECURITY, jsonSchemaPool);
        List<Parameter> headers = getParameterList(securityScheme.headers(), SECURITY, jsonSchemaPool);

        return new PassThroughScheme(queryParameters, headers);
    }

    private static PassThroughScheme buildApiKeySecurityScheme(SecurityScheme securityScheme, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<Parameter> queryParameters = getParameterList(securityScheme.queryParameters(), SECURITY, jsonSchemaPool);
        List<Parameter> headers = getParameterList(securityScheme.headers(), SECURITY, jsonSchemaPool);

        if(securityScheme.settings() instanceof ApiKeySettings){
            String in = ((ApiKeySettings)securityScheme.settings()).in().value();
            String name = ((ApiKeySettings)securityScheme.settings()).name().value();

            if(in.equalsIgnoreCase("query")){
                queryParameters.add(new Parameter(name, SECURITY, buildSimpleStringType(true)));
            }
            else{
                headers.add(new Parameter(name, SECURITY, buildSimpleStringType(true)));
            }
        }

        return new PassThroughScheme(queryParameters, headers);
    }

    //TODO AMF does not support JWT parser to get the security scheme information.
    //     Opened a bug to https://www.mulesoft.org/jira/browse/APIMF-1374
    private static JwtAuthenticationScheme buildJwtAuthenticationScheme(SecurityScheme securityScheme, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<Parameter> queryParameters = getParameterList(securityScheme.queryParameters(), SECURITY, jsonSchemaPool);
        List<Parameter> headers = getParameterList(securityScheme.headers(), SECURITY, jsonSchemaPool);
        List<Parameter> body = new LinkedList<>();
        body.addAll(queryParameters);
        body.addAll(headers);

        return new JwtAuthenticationScheme(headers, body, queryParameters);
    }

    private static CustomAuthenticationScheme buildCustomAuthenticationScheme(SecurityScheme securityScheme, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<Parameter> queryParameters = getParameterList(securityScheme.queryParameters(), SECURITY, jsonSchemaPool);
        List<Parameter> headers = getParameterList(securityScheme.headers(), SECURITY, jsonSchemaPool);

        return new CustomAuthenticationScheme(queryParameters, headers);
    }

    private static OAuth2AuthorizationCodeScheme buildOAuth2AuthorizationCodeSecurityScheme(SecurityScheme securityScheme, OAuth2Flow oAuth2Flow){
        List<String> authorizationGrants = singletonList(OAUTH2_GRANT_AUTHORIZATION_CODE);
        List<String> scopes = oAuth2Flow.scopes().stream().map(s -> s.name().value()).collect(toList());
        String renewTokenExpression = getAnnotatedRenewTokenExpression(securityScheme);

        return new OAuth2AuthorizationCodeScheme(oAuth2Flow.authorizationUri().value(),
                                                 oAuth2Flow.accessTokenUri().value(),
                                                 authorizationGrants,
                                                 scopes,
                                                 renewTokenExpression);
    }

    private static OAuth2ClientCredentialsScheme buildOAuth2ClientCredentialsSecurityScheme(SecurityScheme securityScheme, OAuth2Flow oAuth2Flow){
        List<String> authorizationGrants = singletonList(OAUTH2_GRANT_CLIENT_CREDENTIALS);
        List<String> scopes = oAuth2Flow.scopes().stream().map(s -> s.name().value()).collect(toList());
        String renewTokenExpression = getAnnotatedRenewTokenExpression(securityScheme);

        return new OAuth2ClientCredentialsScheme(oAuth2Flow.authorizationUri().value(),
                                                 oAuth2Flow.accessTokenUri().value(),
                                                 authorizationGrants,
                                                 scopes,
                                                 renewTokenExpression);
    }

    private static String listSchemes(List<SecurityScheme> securitySchemes){
        StringBuilder builder = new StringBuilder();
        for(SecurityScheme securityScheme : securitySchemes){
            builder.append(securityScheme.name());
            builder.append(":");
            builder.append(securityScheme.type().value());
            if(securityScheme.type().value().equals(APISecurityScheme.OAUTH2) && securityScheme.settings() != null){
                OAuth2Settings settings = (OAuth2Settings)securityScheme.settings();
                builder.append(" :");
                if(settings.authorizationGrants() != null && settings.authorizationGrants().size() > 0){
                    for(StrField grant : settings.authorizationGrants()){
                        builder.append(" ");
                        builder.append(grant.value());
                    }
                }
            }
            builder.append(". ");
        }

        return builder.toString();
    }

}
