package org.mule.connectivity.restconnect.internal.modelGeneration.amf.util;

import amf.apicontract.client.platform.model.domain.EndPoint;
import amf.apicontract.client.platform.model.domain.Operation;
import amf.apicontract.client.platform.model.domain.Payload;
import amf.apicontract.client.platform.model.domain.api.WebApi;
import amf.apicontract.client.platform.model.domain.security.SecurityScheme;
import amf.core.client.platform.model.domain.DomainElement;
import amf.core.client.platform.model.domain.DomainExtension;
import amf.core.client.platform.model.domain.ScalarNode;
import amf.core.client.platform.model.domain.Shape;

import org.mule.connectivity.restconnect.exception.GenerationException;
import org.mule.connectivity.restconnect.internal.model.HTTPMethod;
import org.mule.connectivity.restconnect.internal.model.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.model.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.modelGeneration.JsonSchemaPool;
import org.mule.connectivity.restconnect.internal.modelGeneration.util.ParserUtils;

import java.util.LinkedList;
import java.util.List;
import java.util.Optional;

import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.commons.lang3.StringUtils.trimToNull;
import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.AMFTypeDefinitionFactory.getTypeDefinition;

public class AMFParserUtil extends ParserUtils{

    private static final String RC_PREFIX = "rest-connect.";
    private static final String RC_IGNORE_ANNOTATION = RC_PREFIX + IGNORE_ANNOTATION;
    private static final String RC_DEFAULT_ANNOTATION = RC_PREFIX + DEFAULT_ANNOTATION;
    private static final String RC_USER_SELECTED_SECURITY_SCHEMES_ANNOTATION = RC_PREFIX + USER_SELECTED_SECURITY_SCHEMES_ANNOTATION;
    private static final String RC_PARAMETER_NAME_ANNOTATION = RC_PREFIX + PARAMETER_NAME_ANNOTATION;
    private static final String RC_OPERATION_NAME_ANNOTATION = RC_PREFIX + OPERATION_NAME_ANNOTATION;
    private static final String RC_RENEW_TOKEN_EXPRESSION_ANNOTATION = RC_PREFIX + RENEW_TOKEN_EXPRESSION_ANNOTATION;

    private static final String RC_PART_CONTENT_TYPE_ANNOTATION = RC_PREFIX + PART_CONTENT_TYPE_ANNOTATION;
    private static final String RC_PART_FILENAME_ANNOTATION = RC_PREFIX + PART_FILENAME_ANNOTATION;
    private static final String RC_PART_NAME_ANNOTATION = RC_PREFIX + PART_NAME_ANNOTATION;
    private static final String RC_PART_FILENAME_PARAMETER_ANNOTATION = RC_PREFIX + PART_FILENAME_PARAMETER_ANNOTATION;
    private static final String RC_PART_NAME_PARAMETER_ANNOTATION = RC_PREFIX + PART_NAME_PARAMETER_ANNOTATION;

    public static String getMethodDisplayName(Operation operation) {
        String displayName = null;

        if(operation.name().nonEmpty() && !operation.name().value().equalsIgnoreCase(operation.method().value())){
            displayName = operation.name().value();
        }

        return trimToNull(displayName);
    }

    public static String getCanonicalOperationName(EndPoint endPoint, Operation operation, String friendlyName) {
        String annotatedOperationName = getAnnotatedOperationName(operation);

        HTTPMethod httpMethod = HTTPMethod.fromString(operation.method().value());

        return getCanonicalOperationName(httpMethod, endPoint.path().value(), friendlyName, annotatedOperationName);
    }

    public static String getAnnotatedParameterName(amf.apicontract.client.platform.model.domain.Parameter parameter) {
        return getStringAnnotationForElement(parameter, RC_PARAMETER_NAME_ANNOTATION);
    }

    public static String getAnnotatedOperationName(Operation operation) {
        return getStringAnnotationForElement(operation, RC_OPERATION_NAME_ANNOTATION);
    }

    public static String getAnnotatedRenewTokenExpression(SecurityScheme securityScheme) {
        return getStringAnnotationForElement(securityScheme, RC_RENEW_TOKEN_EXPRESSION_ANNOTATION);
    }

    public static String getAnnotatedPartContentType(Shape element) {
        return getStringAnnotationForElement(element, RC_PART_CONTENT_TYPE_ANNOTATION);
    }

    public static String getAnnotatedPartFilename(Shape element) {
        return getStringAnnotationForElement(element, RC_PART_FILENAME_ANNOTATION);
    }

    public static String getAnnotatedPartName(Shape element) {
        return getStringAnnotationForElement(element, RC_PART_NAME_ANNOTATION);
    }

    public static String getAnnotatedPartFilenameParameter(Shape element) {
        return getStringAnnotationForElement(element, RC_PART_FILENAME_PARAMETER_ANNOTATION);
    }

    public static String getAnnotatedPartNameParameter(Shape element) {
        return getStringAnnotationForElement(element, RC_PART_NAME_PARAMETER_ANNOTATION);
    }

    public static String getStringAnnotationForElement(DomainElement element, String annotationName) {
        String stringValue = getAnnotationForElement(element, annotationName);

        if(isNotBlank(stringValue)){
            return stringValue;
        }

        return null;
    }

    public static String getAnnotationForElement(DomainElement element, String annotationName){
        Optional<DomainExtension> optionalDomainExtension = element.customDomainProperties().stream()
                .filter(x -> x.name().nonEmpty() && annotationName.equalsIgnoreCase(x.name().value()))
                .findFirst();

        DomainExtension domainExtension;
        if(optionalDomainExtension.isPresent()){
            domainExtension = optionalDomainExtension.get();
        }
        else{
            return null;
        }

        return ((ScalarNode)domainExtension.extension()).value().value();
    }

    public static boolean elementContainsAnnotation(DomainElement element, String annotationName){
        return element.customDomainProperties().stream().anyMatch(x -> x.name().nonEmpty() && annotationName.equalsIgnoreCase(x.name().value()));
    }

    public static boolean isDefault(Payload payload) {
        if(payload.schema() == null){
            return false;
        }

        return elementContainsAnnotation(payload.schema(), RC_DEFAULT_ANNOTATION);
    }

    public static boolean isIgnored(EndPoint endPoint, Operation operation) {
        return elementContainsAnnotation(endPoint, RC_IGNORE_ANNOTATION)
                || elementContainsAnnotation(operation, RC_IGNORE_ANNOTATION);
    }

    public static boolean isUserSelectedSecuritySchemes(EndPoint endPoint, Operation operation, WebApi api) {
        return elementContainsAnnotation(endPoint, RC_USER_SELECTED_SECURITY_SCHEMES_ANNOTATION)
                || elementContainsAnnotation(operation, RC_USER_SELECTED_SECURITY_SCHEMES_ANNOTATION)
                || elementContainsAnnotation(api, RC_USER_SELECTED_SECURITY_SCHEMES_ANNOTATION);
    }

    public static List<Parameter> getParameterList(List<amf.apicontract.client.platform.model.domain.Parameter> amfParameters, ParameterType parameterType, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<Parameter> parameters = new LinkedList<>();

        for (amf.apicontract.client.platform.model.domain.Parameter amfParameter : amfParameters) {
            parameters.add(new Parameter(amfParameter.parameterName().value(), parameterType, getTypeDefinition(amfParameter, jsonSchemaPool)));
        }

        return parameters;
    }

    public static Shape getActualShape(Shape shape){
        if(shape == null)
            return null;

        if(shape.isLink() && shape.linkTarget().isPresent()){
            return getActualShape((Shape)shape.linkTarget().get());
        }
        return shape;
    }
}
