package org.mule.connectivity.restconnect.internal.model.security;

import static org.apache.commons.lang3.StringUtils.isNotBlank;
import org.mule.connectivity.restconnect.internal.model.typesource.PrimitiveTypeSource;

import java.util.List;

public abstract class OAuth2Scheme extends APISecurityScheme {
    private String authorizationUri;
    private String accessTokenUri;
    private List<String> authorizationGrants;
    private List<String> scopes;
    private String renewTokenExpression =
        "#[attributes.statusCode == 400 or attributes.statusCode == 401 or attributes.statusCode == 402 "
            + "or attributes.statusCode == 403 or attributes.statusCode == 406]";

    public static final String OAUTH2_GRANT_AUTHORIZATION_CODE = "authorization_code";
    public static final String OAUTH2_GRANT_CLIENT_CREDENTIALS = "client_credentials";

    public OAuth2Scheme(String authorizationUri, String accessTokenUri, List<String> grants, List<String> scopes, String renewTokenExpression) {
        this.authorizationUri = authorizationUri;
        this.authorizationGrants = grants;
        this.accessTokenUri = accessTokenUri;
        this.scopes = scopes;

        if(isNotBlank(renewTokenExpression)){
            this.renewTokenExpression = renewTokenExpression;
        }

        initializeConfigParameters();
    }

    @Override
    public String getSchemeName() {
        return OAUTH2;
    }

    public String getAuthorizationUri() {
        return authorizationUri;
    }

    public String getAccessTokenUri() {
        return accessTokenUri;
    }

    public List<String> getAuthorizationGrants() {
        return authorizationGrants;
    }

    public List<String> getScopes() {
        return scopes;
    }

    public String getRenewTokenExpression() {
        return renewTokenExpression;
    }

    @Override
    protected boolean equalProperties(APISecurityScheme scheme) {
        OAuth2Scheme other = (OAuth2Scheme)scheme;

        return other.getScopes().equals(this.getScopes())
                && equalOrBothNull(other.getAuthorizationUri(), this.getAuthorizationUri())
                && equalOrBothNull(other.getAccessTokenUri(), this.getAccessTokenUri())
                && other.getAuthorizationGrants().equals(this.getAuthorizationGrants());
    }

    protected void addScopeConfigParameter() {
        if(getScopes() != null && getScopes().size() > 0){
            StringBuilder scopeBuilder = new StringBuilder();
            for(String scope : getScopes()){
                scopeBuilder.append(scope);
                scopeBuilder.append(" ");
            }
            configParameters.add(getSecuritySchemeParameter("scopes", "scopes", PrimitiveTypeSource.PrimitiveType.STRING, "List of requested scopes separated by spaces", scopeBuilder.toString(), scopeBuilder.toString(), true,false));
        }
    }

    protected boolean equalOrBothNull(Object a, Object b){
        if(a == null){
            return b == null;
        }

        return a.equals(b);
    }
}
