package org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser;

import org.apache.logging.log4j.Logger;
import org.mule.connectivity.restconnect.api.SpecFormat;
import org.mule.connectivity.restconnect.exception.InvalidSourceException;
import org.mule.connectivity.restconnect.internal.model.api.RestConnectModelBuilder;
import org.mule.connectivity.restconnect.internal.model.operation.Operation;
import org.mule.connectivity.restconnect.internal.model.uri.BaseUri;
import org.mule.connectivity.restconnect.internal.model.uri.BaseUriBuilder;
import org.mule.connectivity.restconnect.internal.modelGeneration.JsonSchemaPool;
import org.mule.connectivity.restconnect.internal.modelGeneration.ModelGenerator;
import org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.resourceLoader.RamlParserExchangeDependencyResourceLoader;
import org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.resourceLoader.SafeFileResourceLoader;
import org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.resourceLoader.SafeProxyResourceLoader;

import org.raml.v2.api.RamlModelBuilder;
import org.raml.v2.api.RamlModelResult;
import org.raml.v2.api.loader.ClassPathResourceLoader;
import org.raml.v2.api.loader.CompositeResourceLoader;
import org.raml.v2.api.loader.FileResourceLoader;
import org.raml.v2.api.loader.RamlUrlResourceLoader;
import org.raml.v2.api.loader.ResourceLoader;
import org.raml.v2.api.loader.UrlResourceLoader;
import org.raml.v2.api.model.v10.api.Api;
import org.raml.v2.api.model.v10.methods.Method;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import static java.lang.System.setProperty;
import static org.apache.commons.io.Charsets.UTF_8;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.commons.lang3.StringUtils.join;
import static org.apache.logging.log4j.LogManager.getLogger;
import static org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.util.RamlOperationMappingUtils.getMethods;
import static org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.util.RamlParserUtils.isIgnored;
import static org.mule.connectivity.restconnect.internal.modelGeneration.util.OperationMappingUtils.disambiguateRepeatedOperations;

public class RamlParserModelGenerator extends ModelGenerator {

    private final static Logger LOGGER = getLogger(RamlParserModelGenerator.class);
    private static final String JAVAX_XML_ACCESS_EXTERNAL_DTD = "javax.xml.accessExternalDTD";
    private static final String JAVAX_XML_ACCESS_INTERNAL_DTD = "javax.xml.accessInternalDTD";
    private static final String JAVAX_XML_ACCESS_EXTERNAL_SCHEMA = "javax.xml.accessExternalSchema";
    private static final String FILE_ACCESS = "file";

    public RamlParserModelGenerator(SpecFormat specFormat) {
        super(specFormat);

        if(!specFormat.equals(SpecFormat.RAML)){
            throw new IllegalArgumentException("RamlParser does not support " + specFormat.getName() + " format");
        }
    }

    public RestConnectModelBuilder generateAPIModel(File raml, String rootDir) throws Exception {
        Api api = getAPIFromRamlFile(raml, rootDir, proxyUri, proxyReferer, safeFileLoader);

        Path rootDirPath = new File(rootDir).toPath().toAbsolutePath();

        JsonSchemaPool jsonSchemaPool = new JsonSchemaPool();

        return RestConnectModelBuilder.createModel()
                .withRootDir(rootDirPath)
                .withApiDescription(api.description() != null ? api.description().value() : "")
                .withApiName(api.title() != null ? api.title().value() : "")
                .withBaseUri(buildBaseUri(api))
                .withOperations(buildOperationsModel(api, jsonSchemaPool));
    }

    private static BaseUri buildBaseUri(Api api) {
        String baseUriString = api.baseUri() != null ? api.baseUri().value() : "";
        String versionString = api.version() != null ? api.version().value() : "";

        return BaseUriBuilder.buildBaseUri(baseUriString, versionString);
    }

    private static List<Operation> buildOperationsModel(Api api, JsonSchemaPool jsonSchemaPool) throws Exception {
        final List<Method> methods = getMethods(api);
        List<Operation> operations = new ArrayList<>();

        for (Method method : methods) {
            if(isIgnored(method))
            {
                LOGGER.warn("Resource ignored: {} {}", method.method(), method.resource().resourcePath());
            }
            else
            {
                operations.add(RamlParserOperationGenerator.generateOperation(api, method, jsonSchemaPool));
            }
        }

        disambiguateRepeatedOperations(operations);
        return operations;
    }

    public static Api getAPIFromRamlFile(File raml, String rootDir) throws IOException {
        return getAPIFromRamlFile(raml, rootDir, null, null, false);
    }

    private static Api getAPIFromRamlFile(File raml, String rootDir, String proxyUri, String proxyReferer, boolean safeFileLoader) throws IOException {
        setProperty(JAVAX_XML_ACCESS_EXTERNAL_DTD, FILE_ACCESS);
        setProperty(JAVAX_XML_ACCESS_INTERNAL_DTD, FILE_ACCESS);
        setProperty(JAVAX_XML_ACCESS_EXTERNAL_SCHEMA, FILE_ACCESS);

        if(isRaml08(raml.toPath().toAbsolutePath())){
            throw new InvalidSourceException("RAML 0.8 is not supported.");
        }

        final CompositeResourceLoader resourceLoader = getResourceLoader(rootDir, proxyUri, proxyReferer, safeFileLoader);

        final RamlModelBuilder ramlModelBuilder = new RamlModelBuilder(resourceLoader);

        RamlModelResult ramlModelResult;
        try{
            ramlModelResult = ramlModelBuilder.buildApi(raml);
        }
        catch (Exception e){
            throw new InvalidSourceException("Invalid RAML: Error in RAML Parser: " + e.getMessage() + ".", e);
        }

        if(ramlModelResult.hasErrors()) {
            String ramlErrors = join(ramlModelResult.getValidationResults(), ", ");
            throw new InvalidSourceException("Invalid RAML: " + ramlErrors + ".");
        }

        if(ramlModelResult.isVersion08()) {
            throw new InvalidSourceException("RAML 0.8 is not supported.");
        }

        if(ramlModelResult.getApiV10() == null)
            throw new InvalidSourceException("Invalid RAML: the provided source isn't an API definition but a fragment.");

        return ramlModelResult.getApiV10();
    }

    protected static CompositeResourceLoader getResourceLoader(String rootDir, String proxyUri, String proxyReferer, boolean safeFileLoader) {
        List<ResourceLoader> resourceLoaders = new LinkedList<>();

        if(isNotBlank(proxyUri)){
            resourceLoaders.add(new SafeProxyResourceLoader(proxyUri, proxyReferer));
        }
        else{
            resourceLoaders.add(new CompositeResourceLoader(new UrlResourceLoader(), new RamlUrlResourceLoader()));
        }

        if(safeFileLoader){
            resourceLoaders.add(new SafeFileResourceLoader(rootDir));
        }
        else{
            resourceLoaders.add(new CompositeResourceLoader(new ClassPathResourceLoader(), new FileResourceLoader(".")));
        }

        resourceLoaders.add(new RamlParserExchangeDependencyResourceLoader(rootDir));

        ResourceLoader[] loadersArray = new ResourceLoader[resourceLoaders.size()];
        loadersArray = resourceLoaders.toArray(loadersArray);
        return new CompositeResourceLoader(loadersArray);
    }

    private static boolean isRaml08(Path specPath) throws IOException {
        String content = new String(java.nio.file.Files.readAllBytes(specPath), UTF_8);;

        return content.startsWith("#%RAML 0.8");
    }
}
