package org.mule.connectivity.restconnect.internal.templateEngine.decorator.operation;

import static java.lang.Thread.interrupted;
import static java.util.concurrent.Executors.newCachedThreadPool;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static org.apache.commons.lang.StringUtils.isBlank;
import static org.mule.connectivity.restconnect.internal.model.type.TypeDefinitionBuilder.buildStringTypeWithMediaType;
import static org.slf4j.LoggerFactory.getLogger;
import org.mule.connectivity.restconnect.internal.model.operation.Operation;
import org.mule.connectivity.restconnect.internal.model.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.model.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.model.typesource.JsonTypeSource;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.model.ModelDecorator;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.json.api.JsonTypeLoader;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeoutException;

import org.slf4j.Logger;

public abstract class OperationDecorator {

    protected final Operation operation;
    protected final ModelDecorator modelDecorator;
    protected Boolean inputMetadataTooLong = null;
    protected Boolean outputMetadataTooLong = null;

    private final static Logger LOGGER = getLogger(OperationDecorator.class);

    public OperationDecorator(Operation operation, ModelDecorator modelDecorator) {
        this.operation = operation;
        this.modelDecorator = modelDecorator;
    }

    public TypeDefinition getInputMetadata() {
        TypeDefinition inputMetadata = operation.getInputMetadata();

        if(inputMetadataTooLong == null){
            inputMetadataTooLong = isMetadataTooLong(inputMetadata);
        }

        if(inputMetadataTooLong){
            return buildStringTypeWithMediaType(inputMetadata.getMediaType(), inputMetadata.isRequired());
        }
        else{
            return inputMetadata;
        }
    }

    public TypeDefinition getOutputMetadata() {
        TypeDefinition outputMetadata = operation.getOutputMetadata();

        if(outputMetadataTooLong == null){
            outputMetadataTooLong = isMetadataTooLong(outputMetadata);
        }

        if(outputMetadataTooLong){
            return buildStringTypeWithMediaType(outputMetadata.getMediaType(), outputMetadata.isRequired());
        }
        else{
            return outputMetadata;
        }
    }

    private boolean isMetadataTooLong(TypeDefinition metadata){
        if(metadata == null ||
            metadata.getSource() == null  ||
            !(metadata.getSource() instanceof JsonTypeSource) ||
            isBlank(metadata.getSource().getValue())){

            return false;
        }

        final Callable<Boolean> isMetadataLoaded = () -> {
            final JsonTypeLoader jsonTypeLoader = new JsonTypeLoader(metadata.getSource().getValue());
            final Optional<MetadataType> metadataType1 = jsonTypeLoader.load(null);
            final Optional<MetadataType> metadataType2 = jsonTypeLoader.load(null);

            if(metadataType1.isPresent()){
                if(metadataType1.equals(metadataType2)){
                    return true;
                }
            }
            return !interrupted();
        };

        boolean validMetadata = true;
        final ExecutorService executorService = newCachedThreadPool();
        final Future<Boolean> isMetadataLoadedTask = executorService.submit(isMetadataLoaded);

        try {
            isMetadataLoadedTask.get(200, MILLISECONDS);
            LOGGER.debug("[" + modelDecorator.getGAV() + "]Metadata loaded. Length: "
                             + metadata.getSource().getValue().length());
        } catch (InterruptedException | CancellationException | TimeoutException e) {
            LOGGER.warn("[" + modelDecorator.getGAV() + "]Error: Metadata check timed out. Length: "
                            + metadata.getSource().getValue().length());
            validMetadata = false;
        } catch (ExecutionException e) {
            LOGGER.warn("[" + modelDecorator.getGAV() + "]Error: Invalid metadata.");
            validMetadata = false;
        } finally {
            isMetadataLoadedTask.cancel(true);
            executorService.shutdownNow();
        }

        return !validMetadata;
    }

    public List<Parameter> getQueryParameters() {
        return this.operation.getQueryParameters();
    }

    public List<Parameter> getUriParameters() {
        return this.operation.getUriParameters();
    }

    public List<Parameter> getPartParameters() {
        return this.operation.getPartParameters();
    }

    public List<Parameter> getHeaders() {
        return this.operation.getHeaders();
    }

    public List<Parameter> getParameters() {
        return this.operation.getParameters();
    }

    public boolean hasQueryParameters() {
        return !this.getQueryParameters().isEmpty();
    }

    public boolean hasUriParameters() {
        return !this.getUriParameters().isEmpty();
    }

    public boolean hasHeaders() {
        return !this.getHeaders().isEmpty();
    }

    public String getCanonicalName() {
        return operation.getCanonicalName();
    }

    public String getFriendlyName() {
        return operation.getFriendlyName();
    }

    public String getAnnotatedDisplayName() {
        return operation.getAnnotatedDisplayName();
    }

    public String getHttpMethod() {
        return operation.getHttpMethod();
    }

    public String getUri() {
        return operation.getUri();
    }
}
