package org.mule.connectivity.restconnect.internal.model.typesource;

import static org.apache.commons.lang.StringUtils.isNotBlank;

import org.mule.connectivity.restconnect.exception.GenerationException;

import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.Optional;

import javax.xml.XMLConstants;
import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.ws.commons.schema.XmlSchema;
import org.apache.ws.commons.schema.XmlSchemaCollection;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

public class XmlTypeSource implements TypeSource {

    private String value;
    private String elementName;
    private String schemaPath;
    private boolean wasSchemaParsed = false;

    public XmlTypeSource(String schemaContent) {
        this.value = schemaContent;
        this.elementName = null;
        this.schemaPath = null;
    }

    public XmlTypeSource(String schemaContent, String elementName, String schemaPath) {
        this.value = schemaContent;
        this.elementName = elementName;
        this.schemaPath = schemaPath;
    }

    @Override
    public String getValue() {
        return this.value;
    }

    public String getSchemaPath(){
        return this.schemaPath;
    }

    public String getElementName() throws GenerationException {
        if(!wasSchemaParsed && (elementName == null || !elementName.startsWith("{"))) {
            elementName = getElementQName(elementName);
        }

        return elementName;
    }

    /**
     * Parses the schema's Qname for the given elementName, if null, returns the first found (if none returns null)
     * @param elementName
     * @return the elementName's Qname
     * @throws GenerationException
     */
    private String getElementQName(String elementName) throws GenerationException {
        try {
            InputSource inputSource = new InputSource(new StringReader(this.value));
            Document doc = null;
            doc = parseDoPriv(inputSource, getSecureDocumentBuilder(), doc);

            XmlSchemaCollection schemaCol = new XmlSchemaCollection();
            if(isNotBlank(this.schemaPath)) {
                schemaCol.setBaseUri(new File(this.schemaPath).getParent());
            }
            XmlSchema schema = schemaCol.read(doc, inputSource.getSystemId(), null);
            String result = null;
            if(!schema.getElements().keySet().isEmpty() ) {
                if(elementName == null) {
                    result = schema.getElements().keySet().toArray()[0].toString();
                } else {
                    Optional<QName> optional =
                        schema.getElements().keySet().stream().filter(ks -> elementName.equals(ks.getLocalPart())).findFirst();
                    result = optional.isPresent() ? optional.get().toString() : elementName;
                }
            }
            wasSchemaParsed = true;
            return result;

        } catch (SAXException | IOException e) {
            throw new GenerationException("Could not parse XML element.\n " + this.value);
        }
    }

    private Document parseDoPriv(final InputSource inputSource, final DocumentBuilder builder, Document doc)
        throws IOException, SAXException {
        try {
            doc = java.security.AccessController.doPrivileged(
                (PrivilegedExceptionAction<Document>) () -> builder.parse(inputSource));
        } catch (PrivilegedActionException e) {
            Exception exception = e.getException();
            if (exception instanceof IOException) {
                throw (IOException)exception;
            }
            if (exception instanceof SAXException) {
                throw (SAXException)exception;
            }
        }
        return doc;
    }

    private static final String EXTERNAL_GENERAL_ENTITIES_FEATURE = "http://xml.org/sax/features/external-general-entities";
    private static final String EXTERNAL_PARAMETER_ENTITIES_FEATURE = "http://xml.org/sax/features/external-parameter-entities";
    private static final String LOAD_EXTERNAL_DTD_FEATURE = "http://apache.org/xml/features/nonvalidating/load-external-dtd";

    private static DocumentBuilderFactory dbf = null;
    private static DocumentBuilder getSecureDocumentBuilder() throws GenerationException {
        try {
            if(dbf == null){
                dbf = DocumentBuilderFactory.newInstance();
                dbf.setFeature(XMLConstants.FEATURE_SECURE_PROCESSING, Boolean.TRUE);
                dbf.setFeature(EXTERNAL_GENERAL_ENTITIES_FEATURE, false);
                dbf.setFeature(EXTERNAL_PARAMETER_ENTITIES_FEATURE, false);
                dbf.setFeature(LOAD_EXTERNAL_DTD_FEATURE, false);
                dbf.setXIncludeAware(false);
                dbf.setExpandEntityReferences(false);
                dbf.setNamespaceAware(true);
            }

            return dbf.newDocumentBuilder();
        } catch (ParserConfigurationException e) {
            throw new GenerationException("Could not configure document builder", e);
        }
    }
}
