package org.mule.connectivity.restconnect.internal.modelGeneration.amf;


import amf.apicontract.client.platform.model.domain.Parameter;
import amf.apicontract.client.platform.model.domain.Payload;
import amf.apicontract.client.platform.model.domain.Response;
import amf.core.client.platform.config.RenderOptions;
import amf.core.client.platform.model.domain.ArrayNode;
import amf.core.client.platform.model.domain.DataNode;
import amf.core.client.platform.model.domain.PropertyShape;
import amf.core.client.platform.model.domain.ScalarNode;
import amf.core.client.platform.model.domain.Shape;
import amf.shapes.client.platform.model.domain.AnyShape;
import amf.shapes.client.platform.model.domain.ArrayShape;
import amf.shapes.client.platform.model.domain.FileShape;
import amf.shapes.client.platform.model.domain.NilShape;
import amf.shapes.client.platform.model.domain.NodeShape;
import amf.shapes.client.platform.model.domain.ScalarShape;
import amf.shapes.client.platform.model.domain.SchemaShape;
import amf.shapes.client.platform.model.domain.UnionShape;
import amf.xml.internal.transformer.TypeToXmlSchema;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.ws.commons.schema.XmlSchema;
import org.mule.connectivity.restconnect.exception.GenerationException;
import org.mule.connectivity.restconnect.internal.model.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.model.parameter.PartParameter;
import org.mule.connectivity.restconnect.internal.model.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.model.type.TypeDefinitionBuilder;
import org.mule.connectivity.restconnect.internal.model.typesource.JsonTypeSource;
import org.mule.connectivity.restconnect.internal.model.typesource.MultipartTypeSource;
import org.mule.connectivity.restconnect.internal.model.typesource.PrimitiveTypeSource;
import org.mule.connectivity.restconnect.internal.model.typesource.TypeSource;
import org.mule.connectivity.restconnect.internal.model.typesource.XmlTypeSource;
import org.mule.connectivity.restconnect.internal.modelGeneration.JsonSchemaPool;
import org.mule.connectivity.restconnect.internal.modelGeneration.amf.util.AMFParserUtil;

import javax.ws.rs.core.MediaType;
import java.io.StringWriter;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import static amf.shapes.client.platform.ShapesConfiguration.predefined;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON;
import static javax.ws.rs.core.MediaType.APPLICATION_OCTET_STREAM;
import static javax.ws.rs.core.MediaType.APPLICATION_XML;
import static javax.ws.rs.core.MediaType.MULTIPART_FORM_DATA;
import static javax.ws.rs.core.MediaType.TEXT_PLAIN;
import static org.apache.commons.lang.StringUtils.isNotBlank;
import static org.mule.connectivity.restconnect.internal.model.typesource.PrimitiveTypeSource.PrimitiveType.STRING;
import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.PrimitiveMetadataModelFactory.createPrimitiveMetadataModel;
import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.util.AMFParserUtil.getAnnotatedPartContentType;
import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.util.AMFParserUtil.getAnnotatedPartFilename;
import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.util.AMFParserUtil.getAnnotatedPartFilenameParameter;
import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.util.AMFParserUtil.getAnnotatedPartName;
import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.util.AMFParserUtil.getAnnotatedPartNameParameter;

import amf.shapes.client.platform.ShapesConfiguration;
import amf.shapes.client.platform.ShapesElementClient;

public class AMFTypeDefinitionFactory {

    private static final Logger logger = LogManager.getLogger(AMFTypeDefinitionFactory.class);

    private static final RenderOptions RENDER_OPTIONS = new RenderOptions().withDocumentation().withCompactedEmission();
    private static final ShapesConfiguration JSON_EMITTER_CONFIGURATION = predefined().withRenderOptions(RENDER_OPTIONS);
    private static final ShapesElementClient JSON_EMITTER_CLIENT = JSON_EMITTER_CONFIGURATION.elementClient();

    public static TypeDefinition getTypeDefinition(Parameter parameter, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        return getTypeDefinition(parameter, null, jsonSchemaPool);
    }

    public static TypeDefinition getTypeDefinition(Parameter parameter, String mediaType, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        Shape shape = getParameterShape(parameter);

        if(typeIsDefinedWithRAML(shape)) {
            return buildJsonTypeDefinitionForParameter(parameter, mediaType, jsonSchemaPool);
        }

        if(typeIsDefinedWithSchema(shape)) {
            return buildTypeDefinitionForSchemaShapeParameter(parameter, mediaType, jsonSchemaPool);
        }

        if(typeIsEnum(shape)){
            return buildJsonTypeDefinitionForParameter(parameter, mediaType, jsonSchemaPool);
        }

        if (typeIsPrimitive(shape)) {
            return buildTypeDefinitionForParameter(mediaType, parameter, createPrimitiveMetadataModel(shape));
        }

        if(shape instanceof NilShape)
            return null;

        if (typeDefinitionIsNotProvided(shape)) {
            return buildTypeDefinitionForParameter(mediaType, parameter, new PrimitiveTypeSource(STRING));
        }

        // This shouldn't happen as it's already validated that the declaration is supported.
        logger.warn("Type {} cannot be parsed.", shape.getClass());

        return null;
    }

    private static TypeDefinition buildTypeDefinitionForSchemaShapeParameter(Parameter parameter, String mediaType, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        SchemaShape schemaShape = (SchemaShape)AMFParserUtil.getActualShape(parameter.schema());

        if(schemaShape == null && parameter.payloads().size() > 0){
            schemaShape = (SchemaShape)parameter.payloads().stream().filter(x -> x.schema() instanceof SchemaShape).map(Payload::schema).findFirst().orElse(null);
        }

        if(isXmlSchemaShape(schemaShape)){
            return buildXmlTypeDefinitionForParameter(parameter, mediaType);
        }

        return buildJsonTypeDefinitionForParameter(parameter, mediaType, jsonSchemaPool);
    }

    private static boolean isXmlSchemaShape(SchemaShape schemaShape){
        return schemaShape.raw().nonEmpty() && schemaShape.raw().value().trim().startsWith("<");
    }

    public static TypeDefinition buildInputMetadata(amf.apicontract.client.platform.model.domain.Operation operation, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        if(operation.request() != null){
            List<Payload> payloads = operation.request().payloads();

            return getDefaultTypeDefinition(payloads, jsonSchemaPool);
        }
        return null;
    }

    public static TypeDefinition buildOutputMetadata(amf.apicontract.client.platform.model.domain.Operation operation, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        for(Response response : operation.responses()){
            if(response.statusCode().nonEmpty() && response.statusCode().value().startsWith("2")){
                return getDefaultTypeDefinition(response.payloads(), jsonSchemaPool);
            }
        }
        return null;
    }

    private static TypeDefinition getDefaultTypeDefinition(List<Payload> payloads, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        if(payloads.size() > 0){
            Optional<Payload> defaultPayload = payloads.stream().filter(AMFParserUtil::isDefault).findFirst();

            if(defaultPayload.isPresent()){
                return getTypeDefinitionForPayload(defaultPayload.get(), jsonSchemaPool);
            }

            return getTypeDefinitionForPayload(payloads.get(0), jsonSchemaPool);
        }

        return null;
    }

    private static String getNameForPropertyShape(PropertyShape propertyShape){
        return propertyShape.id().substring(propertyShape.id().lastIndexOf("/") + 1);
    }

    private static TypeDefinition getTypeDefinitionForPayload(Payload payload, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        Shape shape = AMFParserUtil.getActualShape(payload.schema());

        if(MULTIPART_FORM_DATA.equalsIgnoreCase(payload.mediaType().value())){
            if(shape instanceof NodeShape){
                NodeShape nodeShape = (NodeShape)shape;
                if(nodeShape.properties().size() > 0){
                    List<PartParameter> parts =
                            nodeShape.properties()
                                     .stream()
                                     .map(x -> getRestConnectPartParameter(x))
                                     .collect(Collectors.toList());

                    XmlSchema xmlSchema = TypeToXmlSchema.transform("root", shape._internal());
                    xmlSchema.setTargetNamespace("http://validationnamespace.raml.org");

                    StringWriter schemaWriter = new StringWriter();
                    xmlSchema.write(schemaWriter);
                    String schemaString = schemaWriter.toString();

                    return getTypeDefinitionBuilderForShape(MULTIPART_FORM_DATA, shape, new MultipartTypeSource(parts, new XmlTypeSource(schemaString)), true).build();
                }
            }
        }

        if(typeIsEnum(shape)){

            String schemaContent = JSON_EMITTER_CLIENT.buildJsonSchema((AnyShape)shape);
            if(isNotBlank(schemaContent)) {
                if (jsonSchemaPool.containsSchema(schemaContent)) {
                    return jsonSchemaPool.getTypeDefinitionForSchema(schemaContent);
                }
                else{
                    JsonTypeSource source = new JsonTypeSource(schemaContent);
                    TypeDefinition typeDefinition = getTypeDefinitionBuilderForShape(getPayloadMediaTypeOrDefault(payload), shape, source, true).build();
                    jsonSchemaPool.putSchemaTypeDefinitionPair(schemaContent, typeDefinition);
                    return typeDefinition;
                }
            }

        }

        if(typeIsDefinedWithRAML(shape)){
            return  getTypeDefinitionFromSchema(payload, jsonSchemaPool, shape);
        }
        if(typeIsComposed(shape)) {
            return getTypeDefinitionFromSchema(payload, jsonSchemaPool, shape);
        }

        if(typeIsDefinedWithSchema(shape)) {
            SchemaShape schemaShape = (SchemaShape)shape;

            if(isXmlSchemaShape(schemaShape)){
                String elementName = schemaShape.annotations().fragmentName().orElse(null);
                TypeSource source = new XmlTypeSource(schemaShape.raw().value(), elementName, schemaShape.location().get());
                return getTypeDefinitionBuilderForShape(getPayloadMediaTypeOrDefault(payload), shape, source, true).build();
            }
            else{
                return getTypeDefinitionFromSchema(payload, jsonSchemaPool, shape);
            }

        }

        if(typeIsPrimitive(shape)){
            return getTypeDefinitionBuilderForShape(getPayloadMediaTypeOrDefault(payload), shape, createPrimitiveMetadataModel(shape), true).build();
        }

        if(shape instanceof NilShape)
            return null;

        if(typeDefinitionIsNotProvided(shape) ){
            return getTypeDefinitionBuilderForShape(getPayloadMediaTypeOrDefault(payload), shape, new PrimitiveTypeSource(STRING), true).build();
        }

        if(shape == null){
            logger.warn("Type 'null' cannot be parsed.");
        }
        else{
            logger.warn("Type {} cannot be parsed.", shape.getClass());
        }

        return null;
    }

    private static TypeDefinition getTypeDefinitionFromSchema(Payload payload, JsonSchemaPool jsonSchemaPool, Shape shape) throws GenerationException {
        String schemaContent = JSON_EMITTER_CLIENT.buildJsonSchema((AnyShape) shape);
        if(isNotBlank(schemaContent)) {
            if (jsonSchemaPool.containsSchema(schemaContent)) {
                return jsonSchemaPool.getTypeDefinitionForSchema(schemaContent);
            }
            else{
                JsonTypeSource source = new JsonTypeSource(schemaContent);
                TypeDefinition typeDefinition = getTypeDefinitionBuilderForShape(getPayloadMediaTypeOrDefault(payload), shape, source, true).build();
                jsonSchemaPool.putSchemaTypeDefinitionPair(schemaContent, typeDefinition);
                return typeDefinition;
            }
        }
        return null;
    }

    private static String getPayloadMediaTypeOrDefault(Payload payload){
        return payload.mediaType().nonEmpty() ?
            payload.mediaType().value() :
            getDefaultMediaType(payload.schema());
    }

    private static String getDefaultMediaType(Shape shape) {
        if(shape instanceof FileShape)
            return APPLICATION_OCTET_STREAM;

        if(typeIsDefinedWithRAML(shape))
            return APPLICATION_JSON;

        if(typeIsDefinedWithSchema(shape)){
            if(isXmlSchemaShape((SchemaShape)shape))
                return APPLICATION_XML;
            else
                return APPLICATION_JSON;
        }

        if(typeIsComposed(shape)) {
            return APPLICATION_JSON;
        }

        return TEXT_PLAIN;
    }

    private static PartParameter getRestConnectPartParameter(PropertyShape x) {
        String name = getNameForPropertyShape(x);
        Shape range = x.range();

        String defaultMediaType = getDefaultMediaType(range);
        TypeDefinition typeDefinition = getTypeDefinitionBuilderForShape(defaultMediaType, range, new PrimitiveTypeSource(STRING), true).build();

        PartParameter part = new PartParameter(name, ParameterType.PART, typeDefinition);

        String annotatedPartNameParameter = getAnnotatedPartNameParameter(range);
        if(org.apache.commons.lang3.StringUtils.isNotBlank(annotatedPartNameParameter)){
            part.setPartNameParameterName(annotatedPartNameParameter);
        }

        String annotatedPartFilenameParameter = getAnnotatedPartFilenameParameter(range);
        if(org.apache.commons.lang3.StringUtils.isNotBlank(annotatedPartFilenameParameter)){
            part.setPartFilenameParameterName(annotatedPartFilenameParameter);
        }

        String annotatedPartName = getAnnotatedPartName(range);
        if(org.apache.commons.lang3.StringUtils.isNotBlank(annotatedPartName)){
            part.setPartName(annotatedPartName);
        }

        String annotatedPartFilename = getAnnotatedPartFilename(range);
        if(org.apache.commons.lang3.StringUtils.isNotBlank(annotatedPartFilename)){
            part.setPartFilename(annotatedPartFilename);
        }

        String annotatedPartContentType = getAnnotatedPartContentType(range);
        if(org.apache.commons.lang3.StringUtils.isNotBlank(annotatedPartContentType)){
            part.setPartContentType(annotatedPartContentType);
        }

        return part;
    }


    private static TypeDefinition buildTypeDefinitionForParameter(String mediaType, Parameter parameter, TypeSource source){
        Shape shape = AMFParserUtil.getActualShape(parameter.schema());

        return getTypeDefinitionBuilderForShape(mediaType, shape, source, parameter.required().value())
                .withDescription(parameter.description().value())
                .withAnnotatedDisplayName(AMFParserUtil.getAnnotatedParameterName(parameter))
                .build();
    }

    private static TypeDefinitionBuilder getTypeDefinitionBuilderForShape(String mediaType, Shape shape, TypeSource source, boolean required){

        TypeDefinitionBuilder builder =  new TypeDefinitionBuilder(mediaType == null ? null : MediaType.valueOf(mediaType), source, required, shape  instanceof ArrayShape, shape instanceof UnionShape);
        if(shape.defaultValueStr().nonEmpty()){
            /**
             * Since arrray type's default value are changed by defaultValueStr() ie an array value of "A, B"
             * is converted to: "- A\n- B\n- C" we perform a clean up
             */
            String defaultValue = shape.defaultValue() instanceof ArrayNode ?
                ((ArrayNode)shape.defaultValue()).members().stream()
                    .map(i -> ((ScalarNode)i).value().value()).collect(Collectors.joining(","))
                : shape.defaultValueStr().value();

            if(shape instanceof ScalarShape && defaultValue.startsWith("\"") && defaultValue.endsWith("\"")){
                defaultValue = defaultValue.substring(1, defaultValue.length() - 1);
            }

            defaultValue = StringEscapeUtils.escapeJava(defaultValue);

            builder.withDefaultValue(defaultValue);
        }

        if(shape instanceof AnyShape){
            AnyShape anyShape = (AnyShape)shape;
            if(!anyShape.examples().isEmpty() && anyShape.examples().get(0).value().nonEmpty()){
                builder.withExample(anyShape.examples().get(0).value().value());
            }

            if(!anyShape.values().isEmpty()){
                builder.withEnumValues(anyShape.values().stream().map(x -> getEnumValue(x)).collect(Collectors.toList()));
            }

        }

        return builder;
    }

    private static String getEnumValue(DataNode x) {
        if(x instanceof ScalarNode){
            return ((ScalarNode)x).value().value();
        }
        throw new IllegalArgumentException("Enum type not supported");
    }

    private static TypeDefinition buildJsonTypeDefinitionForParameter(Parameter parameter, String mediaType, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        Shape shape = getParameterShape(parameter);

        try {
            String schemaContent = JSON_EMITTER_CLIENT.buildJsonSchema((AnyShape)shape);
            if(isNotBlank(schemaContent)) {
                if (jsonSchemaPool.containsSchema(schemaContent)) {
                    return jsonSchemaPool.getTypeDefinitionForSchema(schemaContent);
                }
                else{
                    JsonTypeSource source = new JsonTypeSource(schemaContent);
                    TypeDefinition typeDefinition = getTypeDefinitionBuilderForShape(mediaType, shape, source, parameter.required().value())
                            .withAnnotatedDisplayName(AMFParserUtil.getAnnotatedParameterName(parameter))
                            .build();
                    jsonSchemaPool.putSchemaTypeDefinitionPair(schemaContent, typeDefinition);
                    return typeDefinition;
                }
            }
            else {
                return getTypeDefinitionBuilderForShape(mediaType, shape, new PrimitiveTypeSource(STRING), parameter.required().value()).build();
            }
        }

        catch (RuntimeException exception) {
            logger.warn("Couldn't generate a type definition for " + parameter.name(), exception);

            return getTypeDefinitionBuilderForShape(mediaType, shape, new PrimitiveTypeSource(STRING), parameter.required().value()).build();
        }
    }

    private static Shape getParameterShape(Parameter parameter) {
        Shape shape = AMFParserUtil.getActualShape(parameter.schema());

        if (shape == null && parameter.payloads().size() > 0) {
            shape = parameter.payloads().stream().filter(x -> x.schema() != null).map(Payload::schema).findFirst().orElse(null);
        }
        return shape;
    }

    private static TypeDefinition buildXmlTypeDefinitionForParameter(Parameter parameter,  String mediaType) {
        Shape shape = getParameterShape(parameter);

        try {
            SchemaShape schemaShape = (SchemaShape)shape;
            String elementName = schemaShape.annotations().fragmentName().orElse(null);
            XmlTypeSource source = new XmlTypeSource(schemaShape.raw().value(), elementName, schemaShape.location().get());

            TypeDefinitionBuilder builder = getTypeDefinitionBuilderForShape(mediaType, shape, source, parameter.required().value());

            return builder.build();
        }

        catch (RuntimeException exception) {
            logger.warn("Couldn't generate a type definition for " + parameter.name(), exception);

            return getTypeDefinitionBuilderForShape(mediaType, shape, new PrimitiveTypeSource(STRING), parameter.required().value()).build();
        }
    }

    public static boolean typeIsDefinedWithSchema(Shape shape){
        return shape instanceof SchemaShape;
    }

    private static boolean typeIsEnum(Shape shape) {
        return shape instanceof AnyShape && !shape.values().isEmpty();
    }

    public static boolean typeIsDefinedWithRAML(Shape shape) {
        return shape instanceof NodeShape ||
               shape instanceof ArrayShape ||
               shape instanceof UnionShape;
    }

    public static boolean typeIsPrimitive(Shape shape) {
        return shape instanceof ScalarShape ||
                shape instanceof FileShape;
    }
    public static boolean typeIsComposed(Shape shape) {
        return shape instanceof AnyShape && !shape.and().isEmpty();
    }

    // This is for the case that the type is not defined (for instance, when there is an example but not a definition).
    public static boolean typeDefinitionIsNotProvided(Shape shape) {
        return shape instanceof AnyShape && !typeIsDefinedWithRAML(shape) && !typeIsPrimitive(shape)
            && !typeIsDefinedWithSchema(shape) && !typeIsComposed(shape);
    }

}
