package org.mule.connectivity.restconnect.internal.templateEngine;

import com.google.googlejavaformat.java.Formatter;
import org.apache.commons.lang3.StringUtils;
import org.apache.velocity.VelocityContext;
import org.mule.connectivity.restconnect.exception.GenerationException;
import org.mule.connectivity.restconnect.exception.UnsupportedSecuritySchemeException;
import org.mule.connectivity.restconnect.internal.model.security.*;
import org.mule.connectivity.restconnect.internal.templateEngine.builder.DevKitTemplateEngineBuilder;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.model.DevKitConnectorModelDecorator;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.operation.DevKitOperationDecorator;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.security.devkit.*;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.type.DevKitConnectorTypeDefinitionDecorator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import static org.apache.commons.lang3.StringUtils.join;


public class DevKitTemplateEngine extends TemplateEngine {

    private static final String CONNECTOR = "connector";
    private static final String OPERATION = "operation";
    private static final String BASE_URI = "baseUri";
    private static final String SECURITY_SCHEME = "securityScheme";
    private static final String SUPPORTS_BASIC_AUTH = "SUPPORTS_BASIC_AUTH";
    private static final String SUPPORTS_PASS_THROUGH = "SUPPORTS_PASS_THROUGH";
    private static final String SUPPORTS_CUSTOM_AUTHENTICATION = "SUPPORTS_CUSTOM_AUTHENTICATION";
    private static final String SUPPORTS_OAUTH2 = "SUPPORTS_OAUTH2";
    private static final String SUPPORTS_OAUTH2_CLIENT_CREDENTIALS = "SUPPORTS_OAUTH2_CLIENT_CREDENTIALS";
    private static final String SUPPORTS_OAUTH2_AUTHORIZATION_CODE = "SUPPORTS_OAUTH2_AUTHORIZATION_CODE";

    private static final String CONNECTOR_TEMPLATE_VM = "templates/devkit/ConnectorTemplate.vm";
    private static final String REST_CLIENT_CONFIG_PROVIDER_VM = "templates/devkit/RestClientConfigProvider.vm";
    private static final String ABSTRACT_CONFIG_VM = "templates/devkit/AbstractConfig.vm";
    private static final String POM_VM = "templates/devkit/pom.vm";
    private static final String LICENSE_MD_VM = "templates/devkit/LICENSE.md";

    private static final Path MAIN_DIR = Paths.get("src/main/java/");

    private final DevKitConnectorModelDecorator model;
    private final Path outputDir;
    private final boolean generateProjectFiles;

    private final Logger logger = LoggerFactory.getLogger(DevKitTemplateEngine.class);

    public DevKitTemplateEngine(DevKitTemplateEngineBuilder builder) {
        this.model = new DevKitConnectorModelDecorator(builder.getModel());
        this.generateProjectFiles = builder.getProjectFilesGeneration();
        this.outputDir = builder.getOutputDir();
    }

    @Override
    public void applyTemplates() throws Exception {
        this.createPojos();
        this.createConnector();
    }

    private void createPojos() {
        Path output = outputDir.resolve(MAIN_DIR);
        output.toFile().mkdirs();

        for (DevKitOperationDecorator operation : model.getDecoratedOperations()) {
            try {
                if(operation.getDecoratedInputMetadata() != null && operation.getDecoratedInputMetadata().requiresPojo()) {
                    operation.getDecoratedInputMetadata().generatePojo(output);
                }
            }
            catch (GenerationException e) {
                operation.getDecoratedInputMetadata().setFailedToGeneratePojo(true);
                logger.error(String.format("Error while creating POJO for input metadata at %s operation", operation.getMethodName()), e);
            }

            try {
                if(operation.getDecoratedOutputMetadata() != null && operation.getDecoratedOutputMetadata().requiresPojo()) {
                    operation.getDecoratedOutputMetadata().generatePojo(output);
                }
            }
            catch (GenerationException e) {
                operation.getDecoratedOutputMetadata().setFailedToGeneratePojo(true);
                logger.error(String.format("Error while creating POJO for output metadata at %s operation", operation.getMethodName()), e);
            }

            for(DevKitConnectorTypeDefinitionDecorator parameter : operation.getDecoratedParameters()) {
                try {
                    if(parameter.requiresPojo()) {
                        parameter.generatePojo(output);
                    }
                }
                catch (GenerationException e) {
                    parameter.setFailedToGeneratePojo(true);
                    logger.error(String.format("Error while creating POJO for parameter %s at %s operation", parameter.getExternalName(), operation.getMethodName()), e);
                }
            }
        }
    }

    private void createConnector() throws Exception {
        VelocityContext context = new VelocityContext(velocityToolManager.createContext());
        context.internalPut(CONNECTOR, model);

        if (model.getBaseUri() != null) {
            context.internalPut(BASE_URI, model.getBaseUriAsString());
        }

        Path outputBaseDir = outputDir;
        Path packagePath = Paths.get(model.getBasePackage().replace('.', File.separatorChar));
        Path mainDir = outputBaseDir.resolve(MAIN_DIR).resolve(packagePath);

        for (DevKitSecuritySchemeDecorator securityScheme : model.getSecuritySchemes()) {
            if(securityScheme instanceof DevKitBasicAuthSchemeDecorator){
                context.internalPut(SUPPORTS_BASIC_AUTH, true);
            }
            if(securityScheme instanceof DevKitOAuth2ClientCredentialsSchemeDecorator) {
                context.internalPut(SUPPORTS_OAUTH2, true);
                context.internalPut(SUPPORTS_OAUTH2_CLIENT_CREDENTIALS, true);
            }
            if(securityScheme instanceof DevKitOAuth2AuthorizationCodeSchemeDecorator) {
                context.internalPut(SUPPORTS_OAUTH2, true);
                context.internalPut(SUPPORTS_OAUTH2_AUTHORIZATION_CODE, true);
            }
            if(securityScheme instanceof DevKitPassThroughSchemeDecorator){
                context.internalPut(SUPPORTS_PASS_THROUGH, true);
            }
            if(securityScheme instanceof DevKitCustomAuthenticationSchemeDecorator){
                context.internalPut(SUPPORTS_CUSTOM_AUTHENTICATION, true);
            }

            if(securityScheme instanceof DevKitOAuth2AuthorizationCodeSchemeDecorator && StringUtils.isNotBlank(securityScheme.getConfigNameSufix())){
                continue;
                //DevKit only supports one @OAuth2 annotated security scheme; We have to skip this one.
            }

            context.internalPut(SECURITY_SCHEME, securityScheme);
            applyTemplate(securityScheme.getTemplateLocation(), mainDir.resolve(securityScheme.getConfigName() + ".java"), context);
            context.internalRemove(SECURITY_SCHEME);
        }

        // Fails when none of the provided OAuth2 grant types are supported and there is no Basic Auth alternative.
        if(!(context.internalContainsKey(SUPPORTS_BASIC_AUTH))
                && context.internalContainsKey(SUPPORTS_OAUTH2)
                && !((context.internalContainsKey(SUPPORTS_OAUTH2_CLIENT_CREDENTIALS) ||
                    (context.internalContainsKey(SUPPORTS_OAUTH2_AUTHORIZATION_CODE) ))) ){
            String grants = getGrants(model.getSecuritySchemes().stream().map(DevKitSecuritySchemeDecorator::getApiSecurityScheme).collect(Collectors.toList()));
            throw new UnsupportedSecuritySchemeException(
                    "OAuth 2.0 grant types supported are Client Credentials and Authorization Code ( " +
                            grants + " not supported )");
        }

        applyTemplate(CONNECTOR_TEMPLATE_VM, mainDir.resolve(model.getClassName() + ".java"), context);
        applyTemplate(ABSTRACT_CONFIG_VM, mainDir.resolve("AbstractConfig.java"), context);
        applyTemplate(REST_CLIENT_CONFIG_PROVIDER_VM, mainDir.resolve("RestClientConfigProvider.java"), context);
        applyTemplate(LICENSE_MD_VM, outputBaseDir.resolve("LICENSE.md"), context);

        // Generate pom.xml only when project file generation is enabled.
        if(generateProjectFiles) {
            Path pomPath = outputBaseDir.resolve("pom.xml");

            applyTemplate(POM_VM, pomPath, context);
            checkPomIsConsistent(pomPath);
        }
    }

    @Override
    protected Path applyTemplate(String templateVm, Path output, VelocityContext context) throws Exception {
        Path path = super.applyTemplate(templateVm, output, context);

        if(output.toString().endsWith(".java")){
            javaFormat(output);
        }

        return path;
    }

    private void javaFormat(Path filepath) throws IOException {

        String formattedSource;

        try{
            File file = filepath.toFile();

            String content = new String(Files.readAllBytes(filepath), "UTF-8");
            formattedSource = new Formatter().formatSource(content);

            file.delete();
        }
        catch(Exception e){
            logger.error("Could not format file " + filepath.toString(), e);
            return;
        }

        Files.write(filepath, formattedSource.getBytes("UTF-8"));
    }

    private String getGrants(List<APISecurityScheme> schemas){
        List<String> grants = new ArrayList<>();

        for(APISecurityScheme schema : schemas){
            if(schema instanceof OAuth2Scheme){
                grants.addAll(((OAuth2Scheme) schema).getAuthorizationGrants());
            }
        }

        return join(grants, ", ");
    }

}
