package org.mule.connectivity.restconnect.internal.modelGeneration.amf.resourceLoader;

import static java.lang.String.format;
import static java.util.concurrent.CompletableFuture.runAsync;
import static org.mule.connectivity.restconnect.internal.modelGeneration.util.ResourceLoaderUtils.getHttpURLConnection;

import java.net.HttpURLConnection;
import java.net.URL;
import java.util.concurrent.CompletableFuture;

import amf.core.client.common.remote.Content;
import amf.core.client.platform.resource.BaseHttpResourceLoader;
import amf.core.internal.remote.UnexpectedStatusCode;
import scala.Option;
import org.codehaus.plexus.util.IOUtil;


public class SafeProxyResourceLoader extends BaseHttpResourceLoader {

  private static final String USER_AGENT = "rest-connect";

  private final String proxyBaseUri;
  private final String referer;
  private final String token;
  private final String orgId;

  public SafeProxyResourceLoader(String proxyBaseUri, String referer, String token, String orgId){
    this.proxyBaseUri = proxyBaseUri.endsWith("/") ? proxyBaseUri.substring(0, proxyBaseUri.length()-1) : proxyBaseUri;
    this.referer = referer;
    this.token = token;
    this.orgId = orgId;
  }

  @Override
  public CompletableFuture<Content> fetch(String resource) {
    if (resource == null || resource.isEmpty()){
      return fail();
    }

    CompletableFuture<Content> f = new CompletableFuture<>();

    runAsync(() -> {
      try{
        URL u = new URL(format("%s/%s", proxyBaseUri, resource));
        HttpURLConnection connection = getHttpURLConnection(u, referer, token, orgId);

        Content content = createContent(connection);
        f.complete(content);
      }
      catch (Exception e){
        f.completeExceptionally(e);
      }

      f.completeExceptionally(new Exception("Could not complete future"));
    });

    return f;
  }

  private Content createContent(final HttpURLConnection connection) throws Exception {
    connection.connect();
    int statusCode = connection.getResponseCode();

    if(statusCode == 200){
      Option<String> option = Option.apply(connection.getHeaderField("Content-Type"));
      return new Content(IOUtil.toString(connection.getInputStream()), connection.getURL().toString(), option);
    }
    else{
      throw new UnexpectedStatusCode(connection.getURL().toString(), statusCode);
    }
  }

  private CompletableFuture<Content> fail() {
    return CompletableFuture.supplyAsync(() -> {
      throw new RuntimeException("Failed to apply.");
    });
  }
}
