package org.mule.connectivity.restconnect.internal.model.api;

import org.mule.connectivity.restconnect.internal.model.ConnectorCategory;
import org.mule.connectivity.restconnect.internal.model.MavenDependency;
import org.mule.connectivity.restconnect.internal.model.operation.Operation;
import org.mule.connectivity.restconnect.internal.model.uri.BaseUri;
import org.mule.connectivity.restconnect.internal.modelGeneration.util.ParserUtils;

import java.nio.file.Path;
import java.util.List;

import static org.apache.commons.lang.StringUtils.isBlank;


public class RestConnectAPIModel {
    public final static String DEFAULT_VERSION = "1.0.0";

    private final String apiName;
    private final String description;
    private final BaseUri baseUri;
    private final ConnectorCategory category;
    private final String projectDescription;
    private final List<MavenDependency> dependencies;
    private final List<Operation> operations;

    private String groupId;
    private String artifactId;
    private String version;
    private String customPackage;
    private Path rootDir;

    private boolean tlsEnabled = false;

    public RestConnectAPIModel(RestConnectModelBuilder builder) {
        this.apiName = buildApiName(builder);
        this.description = buildApiDescription(builder);

        this.groupId = buildGroupId(builder);
        this.artifactId = buildArtifactId(builder);
        this.version = buildVersion(builder);

        this.customPackage = buildCustomPackage(builder);

        this.category = buildCategory(builder);
        this.dependencies = buildDependencies(builder);
        this.projectDescription = buildProjectDescription(builder);

        this.baseUri = buildBaseUri(builder);
        this.operations = buildOperations(builder);
        this.tlsEnabled = buildTlsEnabled(builder);

        this.rootDir = builder.getRootDir();
    }

    private List<Operation> buildOperations(RestConnectModelBuilder builder) {
        return builder.getOperations();
    }
    private boolean buildTlsEnabled(RestConnectModelBuilder builder) { return builder.isTlsEnabled(); }

    private BaseUri buildBaseUri(RestConnectModelBuilder builder) {
        return builder.getBaseUri();
    }

    private String buildProjectDescription(RestConnectModelBuilder builder) {
        return builder.getProjectDescription();
    }

    private List<MavenDependency> buildDependencies(RestConnectModelBuilder builder) {
        return builder.getDependencies();
    }

    private ConnectorCategory buildCategory(RestConnectModelBuilder builder) {
        return builder.getCategory();
    }

    private String buildApiName(RestConnectModelBuilder builder) {
        return builder.getApiName();
    }

    private String buildApiDescription(RestConnectModelBuilder builder) {
        return builder.getApiDescription();
    }

    private String buildVersion(RestConnectModelBuilder builder) {
        if(!isBlank(builder.getVersion()))
            return ParserUtils.removeMavenVersionUnwantedCharacters(builder.getVersion());

        return getDefaultVersion();
    }

    private String buildGroupId(RestConnectModelBuilder builder) {
        if(!isBlank(builder.getGroupId()))
            return ParserUtils.removeMavenGroupUnwantedCharacters(builder.getGroupId());

        return null;
    }

    private String buildArtifactId(RestConnectModelBuilder builder) {
        if(!isBlank(builder.getArtifactId()))
            return ParserUtils.removeMavenArtifactUnwantedCharacters(builder.getArtifactId());

        return null;
    }

    private String buildCustomPackage(RestConnectModelBuilder builder) {
        if(!isBlank(builder.getCustomPackage()))
            return builder.getCustomPackage();

        return null;
    }

    public String getApiName() {
        return apiName;
    }

    protected String getDefaultVersion() {
        return DEFAULT_VERSION;
    }

    public List<Operation> getOperations() {
        return operations;
    }

    public String getGroupId() {
        return groupId;
    }

    public String getArtifactId() {
        return artifactId;
    }

    public String getVersion() {
        return version;
    }

    public String getCustomPackage() {
        return customPackage;
    }

    public ConnectorCategory getCategory() {
        return category;
    }

    public List<MavenDependency> getDependencies() {
        return dependencies;
    }

    public BaseUri getBaseUri() {
        return baseUri;
    }

    public String getProjectDescription() {
        return projectDescription;
    }

    public String getApiDescription() {
        return description;
    }

    public boolean isTlsEnabled() { return tlsEnabled; }

    public Path getRootDir() {
        return rootDir;
    }
}
