package org.mule.connectivity.restconnect.internal.model.api;

import org.apache.commons.lang3.StringUtils;
import org.mule.connectivity.restconnect.internal.model.ConnectorCategory;
import org.mule.connectivity.restconnect.internal.model.MavenDependency;
import org.mule.connectivity.restconnect.internal.model.operation.Operation;
import org.mule.connectivity.restconnect.internal.model.uri.BaseUri;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

public class RestConnectModelBuilder {

    private String apiName;
    private String apiDescription;

    private String groupId;
    private String artifactId;
    private String version;
    private String customPackage;
    private ConnectorCategory category;
    private List<MavenDependency> dependencies;
    private String projectDescription;
    private List<Operation> operations;
    private BaseUri baseUri;
    private Path rootDir;

    private boolean tlsEnabled = false;

    private RestConnectModelBuilder() {
        this.dependencies = new ArrayList<>();
    }

    public static RestConnectModelBuilder createModel() {
        return new RestConnectModelBuilder();
    }

    public RestConnectModelBuilder withApiName(String apiName) {
        if(StringUtils.isNotBlank(apiName)){
            this.apiName = apiName;
        }

        return this;
    }

    public RestConnectModelBuilder withGroupId(String groupId) {
        this.groupId = groupId;
        return this;
    }

    public RestConnectModelBuilder withArtifactId(String artifactId) {
        this.artifactId = artifactId;
        return this;
    }

    public RestConnectModelBuilder withVersion(String version) {
        this.version = version;
        return this;
    }

    public RestConnectModelBuilder withCategory(ConnectorCategory category) {
        this.category = category;
        return this;
    }

    public RestConnectModelBuilder withCustomPackage(String customPackage) {
        this.customPackage = customPackage;
        return this;
    }

    public RestConnectModelBuilder withCategory(String category) {
        this.category = ConnectorCategory.fromString(category);
        return this;
    }

    public RestConnectModelBuilder withDependency(MavenDependency dependency) {
        this.dependencies.add(dependency);
        return this;
    }

    public RestConnectModelBuilder withProjectDescription(String description) {
        this.projectDescription = description;
        return this;
    }

    public RestConnectModelBuilder withApiDescription(String apiDescription) {
        this.apiDescription = apiDescription;
        return this;
    }

    public RestConnectModelBuilder withOperations(List<Operation> operations) {
        this.operations = operations;
        return this;
    }

    public RestConnectModelBuilder withBaseUri(BaseUri baseUri) {
        this.baseUri = baseUri;
        return this;
    }

    public RestConnectModelBuilder withTlsEnabled(boolean tlsEnabled) {
        this.tlsEnabled = tlsEnabled;
        return this;
    }

    public boolean isTlsEnabled() {
        return tlsEnabled;
    }

    public String getApiName() {
        return apiName;
    }

    public String getGroupId() {
        return groupId;
    }

    public String getArtifactId() {
        return artifactId;
    }

    public String getVersion() {
        return version;
    }

    public String getCustomPackage() {
        return customPackage;
    }

    public List<MavenDependency> getDependencies() {
        return dependencies;
    }

    public String getProjectDescription() {
        return projectDescription;
    }

    public ConnectorCategory getCategory() {
        return category;
    }

    public BaseUri getBaseUri() {
        return baseUri;
    }

    public List<Operation> getOperations() {
        return operations;
    }

    public String getApiDescription() {
        return apiDescription;
    }

    public RestConnectAPIModel build() {
        return new RestConnectAPIModel(this);
    }

    public RestConnectModelBuilder withRootDir(Path rootDir) {
        this.rootDir = rootDir;
        return this;
    }

    public Path getRootDir() {
        return this.rootDir;
    }
}
