package org.mule.connectivity.restconnect.internal.templateEngine.decorator.model;

import com.google.common.collect.ImmutableList;
import org.apache.commons.lang3.StringUtils;
import org.mule.connectivity.restconnect.internal.model.api.RestConnectAPIModel;
import org.mule.connectivity.restconnect.internal.model.operation.Operation;
import org.mule.connectivity.restconnect.internal.model.security.APISecurityScheme;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.operation.SmartConnectorOperationDecorator;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.security.smartconnector.SmartConnectorMergedSchemeDecorator;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.security.smartconnector.SmartConnectorSecuritySchemeDecorator;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.type.SmartConnectorTypeDefinitionDecorator;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;

import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.connectivity.restconnect.internal.modelGeneration.util.ParserUtils.getXmlName;
import static org.mule.connectivity.restconnect.internal.modelGeneration.util.ParserUtils.removeMavenArtifactUnwantedCharacters;


public class SmartConnectorModelDecorator extends ModelDecorator {

    public static final String DEFAULT_GROUP_ID = "org.mule.modules.extensions";
    public static final String DEFAULT_ARTIFACT_PREFIX = "mule-module-";
    private final List<SmartConnectorOperationDecorator> decoratedOperations;
    private final HashMap<APISecurityScheme, String> securitySchemesNames = new HashMap<>();
    private final HashMap<String, SmartConnectorTypeDefinitionDecorator> allTypeDecorators = new HashMap<>();
    private final SmartConnectorMergedSchemeDecorator mergedScheme;
    private final boolean hasOauthScheme;

    private final boolean tlsEnabled;

    public SmartConnectorModelDecorator(RestConnectAPIModel model) {
        super(model);
        this.tlsEnabled = model.isTlsEnabled();
        this.decoratedOperations = buildDecoratedOperations();
        this.mergedScheme = buildMergedSecurityScheme();
        this.hasOauthScheme = securitySchemesNames.keySet().stream().anyMatch(x -> x.getSchemeName().equals(APISecurityScheme.OAUTH2));
    }

    private List<SmartConnectorOperationDecorator> buildDecoratedOperations() {
        ImmutableList.Builder<SmartConnectorOperationDecorator> builder = ImmutableList.builder();

        for (Operation operation : this.getOperations()) {
            builder.add(new SmartConnectorOperationDecorator(operation, securitySchemesNames, allTypeDecorators, this));
        }

        return builder.build();
    }

    public List<SmartConnectorTypeDefinitionDecorator> getAllParameterDecorators(){
        return new ArrayList<>(allTypeDecorators.values());
    }

    @Override
    public String getGroupId() {
        return isNotBlank(super.getGroupId()) ? super.getGroupId() : DEFAULT_GROUP_ID;
    }

    @Override
    public String getArtifactId() {
        return isNotBlank(super.getArtifactId()) ? super.getArtifactId() : getDefaultArtifactId();
    }

    protected List<SmartConnectorSecuritySchemeDecorator> getOperationsSecuritySchemes() {
        List<SmartConnectorSecuritySchemeDecorator> operationSchemes = new LinkedList<>();

        for(SmartConnectorOperationDecorator operation : getDecoratedOperations()){

            if(operation.isUserSelectedSecuritySchemes()){
                for(SmartConnectorSecuritySchemeDecorator operationSecurityScheme : operation.getAllSecuritySchemes()){
                    if(operationSchemes.stream().noneMatch(x -> x.equals(operationSecurityScheme))){
                        operationSchemes.add(operationSecurityScheme);
                    }
                }
            }
            else{
                SmartConnectorSecuritySchemeDecorator operationSecurityScheme = operation.getSecurityScheme();

                if(operationSchemes.stream().noneMatch(x -> x.equals(operationSecurityScheme))){
                    operationSchemes.add(operationSecurityScheme);
                }
            }
        }

        return operationSchemes;
    }

    public SmartConnectorMergedSchemeDecorator buildMergedSecurityScheme() {
        return new SmartConnectorMergedSchemeDecorator(getOperationsSecuritySchemes(), getModulePrefix(), tlsEnabled);
    }

    public SmartConnectorMergedSchemeDecorator getMergedSecurityScheme() {
        return mergedScheme;
    }

    private String getDefaultArtifactId() {
        return DEFAULT_ARTIFACT_PREFIX + getModulePrefix();
    }

    public String getModulePrefix() {
        return removeMavenArtifactUnwantedCharacters(getXmlName(super.getApiName()));
    }

    public String getNamespace() {
        return "http://www.mulesoft.org/schema/mule/" + getModulePrefix();
    }

    public String getBasePath() {
        if(getBaseUri() == null)
            return "/";

        return StringUtils.isBlank(getBaseUri().getPath()) ? "/" : getBaseUri().getPath();
    }

    public String getProtocol() {
        if(getBaseUri() == null)
            return null;

        return getBaseUri().getScheme().toUpperCase();
    }

    public String getHost() {
        if(getBaseUri() == null)
            return null;

        return getBaseUri().getHost();
    }

    public Integer getPort() {
        if(getBaseUri() == null)
            return null;

        if(getBaseUri().getPort() != -1)
            return getBaseUri().getPort();

        switch (getProtocol()) {
            case "HTTP": return 80;
            case "HTTPS": return 443;
        }

        return null;
    }

    public List<SmartConnectorOperationDecorator> getDecoratedOperations() {
        return this.decoratedOperations;
    }

    public String getProjectDescription() {
        String projectDescription = model.getProjectDescription();
        String apiDescription = model.getApiDescription();

        String finalDescription = StringUtils.EMPTY;

        if(isNotBlank(projectDescription) && isNotBlank(apiDescription)){
            finalDescription = format("%s - %s", projectDescription, apiDescription);
        }
        else if(isNotBlank(projectDescription)){
            finalDescription = projectDescription;
        }
        else if(isNotBlank(apiDescription)){
            finalDescription = apiDescription;
        }

        return StringUtils.abbreviate(finalDescription, 250);
    }

    public boolean hasOauthScheme() {
        return hasOauthScheme;
    }

    public boolean isTlsEnabled()  { return tlsEnabled; }
}
