package org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser;

import org.apache.commons.lang3.StringUtils;
import org.mule.connectivity.restconnect.internal.model.HTTPMethod;
import org.mule.connectivity.restconnect.internal.model.operation.Operation;
import org.mule.connectivity.restconnect.internal.model.operation.OperationBuilder;
import org.mule.connectivity.restconnect.internal.model.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.model.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.model.security.APISecurityScheme;
import org.mule.connectivity.restconnect.internal.modelGeneration.JsonSchemaPool;
import org.raml.v2.api.model.v10.api.Api;
import org.raml.v2.api.model.v10.datamodel.ObjectTypeDeclaration;
import org.raml.v2.api.model.v10.datamodel.UnionTypeDeclaration;
import org.raml.v2.api.model.v10.methods.Method;
import org.raml.v2.api.model.v10.resources.Resource;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import static org.mule.connectivity.restconnect.internal.model.parameter.ParameterType.QUERY;
import static org.mule.connectivity.restconnect.internal.model.type.TypeDefinitionBuilder.buildSimpleStringType;
import static org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.security.RamlParserSecuritySchemeFactory.getOperationSecuritySchemes;
import static org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.util.RamlParserUtils.getAnnotatedOperationName;
import static org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.util.RamlParserUtils.getCanonicalOperationName;
import static org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.util.RamlParserUtils.getMethodDescription;
import static org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.util.RamlParserUtils.getMethodDisplayName;
import static org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.util.RamlParserUtils.getParameterList;
import static org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.util.RamlParserUtils.isGenerateUserSelectedSecuritySchemes;

public class RamlParserOperationGenerator {

    public static Operation generateOperation(Api api, Method method, JsonSchemaPool jsonSchemaPool) throws Exception {

        RamlParserTypeDefinitionFactory typeDefinitionFactory = new RamlParserTypeDefinitionFactory(method, jsonSchemaPool);

        String friendlyName = getMethodDisplayName(method);
        String annotatedName = getAnnotatedOperationName(method);

        Operation operation = new OperationBuilder()
                .withFriendlyName(friendlyName)
                .withAnnotatedDisplayName(StringUtils.isNotBlank(annotatedName) ? annotatedName : friendlyName)
                .withCanonicalName(getCanonicalOperationName(method, getMethodDisplayName(method)))
                .withDescription(getMethodDescription(method))
                .withHttpMethod(HTTPMethod.fromString(method.method()))
                .withUri(method.resource().resourcePath())
                .withUriParams(buildUriParams(method, jsonSchemaPool))
                .withQueryParams(buildQueryParams(method, jsonSchemaPool))
                .withHeaders(buildHeaders(method, jsonSchemaPool))
                .withInputMetadata(typeDefinitionFactory.constructInputMetadata())
                .withOutputMetadata(typeDefinitionFactory.constructOutputMetadata())
                .withSecuritySchemes(buildSecuritySchemes(api, method, jsonSchemaPool))
                .withUserSelectedSecuritySchemes(isGenerateUserSelectedSecuritySchemes(method, api))
                .build();

        return operation;
    }

    private static List<APISecurityScheme> buildSecuritySchemes(Api api, Method method, JsonSchemaPool jsonSchemaPool) throws Exception {
        return getOperationSecuritySchemes(api, method, jsonSchemaPool);
    }

    private static List<Parameter> buildQueryParams(Method method, JsonSchemaPool jsonSchemaPool) throws Exception {
        if(method.queryString() != null) {
            return buildQueryParametersFromQueryString(method);

        }
        return getParameterList(method.queryParameters(), QUERY, jsonSchemaPool);
    }

    private static List<Parameter> buildQueryParametersFromQueryString(Method method) {
        Set<String> parameters = new HashSet<>();
        if(method.queryString() instanceof ObjectTypeDeclaration) {
            ((ObjectTypeDeclaration) method.queryString()).properties().stream()
                .forEach(p -> parameters.add(p.name()));
        } else if (method.queryString() instanceof UnionTypeDeclaration) {
           ((UnionTypeDeclaration) method.queryString()).of().stream()
               .forEach(typeDeclaration -> ((ObjectTypeDeclaration)typeDeclaration).properties().stream()
                   .forEach(p -> parameters.add(p.name())));
        }
        return new ArrayList<>(parameters).stream()
            .map(p -> new Parameter(p, QUERY, buildSimpleStringType(false)))
            .collect(Collectors.toList());
    }

    private static List<Parameter> buildUriParams(Method method, JsonSchemaPool jsonSchemaPool) throws Exception {
        List<Parameter> uriParameters = new ArrayList<>();

        // Due to RESTC-34, this is done recursively to the top.
        Resource resource = method.resource();

        while(resource != null) {

            for(Parameter uriParam : getParameterList(resource.uriParameters(), ParameterType.URI, jsonSchemaPool)){
                if(uriParameters.stream().noneMatch(y -> y.getExternalName().equalsIgnoreCase(uriParam.getExternalName()))){
                    uriParameters.add(0, uriParam);
                }
            }

            resource = resource.parentResource();
        }

        return uriParameters;
    }

    private static List<Parameter> buildHeaders(Method method, JsonSchemaPool jsonSchemaPool) throws Exception {
        return getParameterList(method.headers(), ParameterType.HEADER, jsonSchemaPool);
    }

}
