package org.mule.connectivity.restconnect.internal.util;

import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.mule.connectivity.restconnect.internal.model.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.model.typesource.MultipartTypeSource;
import org.mule.connectivity.restconnect.internal.model.typesource.TypeSource;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.type.SmartConnectorTypeDefinitionDecorator;

import java.util.ArrayList;
import java.util.List;

import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.join;
import static org.mule.connectivity.restconnect.internal.modelGeneration.util.ParserUtils.getXmlName;

public class DataWeaveUtils {

    public static String getParametersDW(final List<Parameter> parameters, final List<Parameter> properties) {
        List<String> keys = new ArrayList<>();

        if(parameters != null){
            for (Parameter param : parameters) {
                keys.add(getParameterDWExpression(param));
            }
        }

        if(properties != null){
            for (Parameter prop : properties) {
                keys.add(getParameterDWExpression(prop, "property", "_"));
            }
        }

        return format("#[ { %s } filterObject $ != null ]", join(keys, ","));
    }

    private static String getParameterDWExpression(Parameter parameter){
        return getParameterDWExpression(parameter, "", "");
    }

    private static String getParameterDWExpression(Parameter parameter, String prefix, String prefixSeparator){

        if(parameter.hasFixedValue()){
            return format("'%s': '%s'", escapeDataWeave(parameter.getExternalName()), parameter.getFixedValue());
        }

        String xmlName = getXmlName(parameter.getInternalName());
        String prefixedName = (StringUtils.isNotBlank(prefix) ? prefix + (xmlName.startsWith(prefixSeparator) ? "" : prefixSeparator) : "") + xmlName;

        return format("'%s': vars['%s']", escapeDataWeave(parameter.getExternalName()), prefixedName);
    }

    private static String escapeDataWeave(String string){
        return StringEscapeUtils.escapeJava(string).replace("$", "\\$");
    }

    private static String getBodyDWExpression(String mediaType, String metadataName) {
        if (mediaType.equals("application/x-www-form-urlencoded")) {
            StringBuilder stringBuilder = new StringBuilder();
            stringBuilder.append("#[output ").append(mediaType).append(System.lineSeparator())
                    .append("fun formUrlEncoded(obj: Object) = obj mapObject {").append(System.lineSeparator())
                    .append("($$) : $ match {").append(System.lineSeparator())
                    .append("case x is Object -> write(x, \"application/json\")").append(System.lineSeparator())
                    .append("case x is Array -> write(x, \"application/json\")").append(System.lineSeparator())
                    .append("else -> $").append(System.lineSeparator())
                    .append("}}").append(System.lineSeparator())
                    .append("---").append(System.lineSeparator())
                    .append("formUrlEncoded(vars['").append(metadataName).append("'])]").append(System.lineSeparator());
            return stringBuilder.toString();
        } else {
            return format("#[output %s --- vars['%s']]%s", mediaType, metadataName, System.lineSeparator());
        }
    }

    public static String getInputPayloadDWExpression(SmartConnectorTypeDefinitionDecorator typeDefinition){
        TypeSource source = typeDefinition.getTypeDefinition().getSource();
        StringBuilder strBuilder = new StringBuilder();

        if(source instanceof MultipartTypeSource){
            MultipartTypeSource multiPartSource = (MultipartTypeSource)source;
            strBuilder.append("#[").append(System.lineSeparator());
            strBuilder.append("%dw 2.0").append(System.lineSeparator());
            strBuilder.append("output multipart/form-data").append(System.lineSeparator());
            strBuilder.append("boundary='rc34b21aa'").append(System.lineSeparator());
            strBuilder.append("---").append(System.lineSeparator());
            strBuilder.append("parts : {").append(System.lineSeparator());
            String prefix = "";
            for(Parameter param : multiPartSource.getParts()){
                //If it's not the first part, add a comma.
                strBuilder.append(prefix).append(System.lineSeparator());
                prefix = ",";

                strBuilder.append(escapeDataWeave(param.getExternalName())).append(" : {").append(System.lineSeparator());
                strBuilder.append("headers : {").append(System.lineSeparator());
                strBuilder.append("\"Content-Type\": \"").append(param.getTypeDefinition().getMediaType()).append("\"").append(System.lineSeparator());
                strBuilder.append("},").append(System.lineSeparator());
                strBuilder.append("content : ").append("vars['").append(typeDefinition.getInternalName()).append("']");
                if(typeDefinition.hasElementName() && StringUtils.isNotEmpty(typeDefinition.getElementName())){
                    String elementName = typeDefinition.getElementName().replaceAll("\\{.*\\}", "");
                    strBuilder.append("['").append(elementName).append("']");
                }
                strBuilder.append("['").append(param.getInternalName()).append("']");

                strBuilder.append("}");
            }
            strBuilder.append("}").append(System.lineSeparator());
            strBuilder.append("]");
        }
        else{
            return getBodyDWExpression(typeDefinition.getMediaTypeForTransformation(), typeDefinition.getInternalName());
        }

        return strBuilder.toString();
    }
}
