package org.mule.connectivity.restconnect.internal.model.security;

import org.mule.connectivity.restconnect.internal.model.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.model.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.model.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.model.type.TypeDefinitionBuilder;
import org.mule.connectivity.restconnect.internal.model.typesource.PrimitiveTypeSource;

import java.util.LinkedList;
import java.util.List;

public abstract class APISecurityScheme{

    public static final String BASIC = "Basic Authentication";
    public static final String OAUTH1 = "OAuth 1.0"; // Not supported
    public static final String OAUTH2 = "OAuth 2.0";
    public static final String PASS_THROUGH = "Pass Through";
    public static final String DIGEST_AUTHENTICATION = "Digest Authentication";
    public static final String CUSTOM_AUTHENTICATION = "Custom Authentication";
    public static final String UNSECURED = "";

    protected List<Parameter> configParameters = new LinkedList<>();
    protected List<Parameter> headers = new LinkedList<>();
    protected List<Parameter> queryParameters = new LinkedList<>();

    public APISecurityScheme(){
        initializeConfigParameters();
    }

    public abstract String getSchemeName();

    public abstract void initializeConfigParameters();

    public List<Parameter> getConfigParameters(){
        return configParameters;
    }

    public List<Parameter> getQueryParameters(){
        return queryParameters;
    }

    public List<Parameter> getHeaders(){
        return headers;
    }

    public boolean hasQueryParameters(){
        return !queryParameters.isEmpty();
    }

    public boolean hasHeaders(){
        return !headers.isEmpty();
    }

    public boolean hasConfigParameters(){
        return !configParameters.isEmpty();
    }

    public boolean equals(APISecurityScheme scheme){
        return
                scheme.getClass().equals(this.getClass())
                && scheme.getSchemeName().equals(this.getSchemeName())
                && equalProperties(scheme);
    }

    //This method should check if the properties defined for the security schemes are equal
    protected abstract boolean equalProperties(APISecurityScheme scheme);

    public static Parameter getSecuritySchemeParameter(String name, String displayName, PrimitiveTypeSource.PrimitiveType type, String description, String defaultValue, String example, boolean required, boolean isPassword){
        TypeDefinition typeDefinition = TypeDefinitionBuilder.buildSimplePrimitiveType(type, required);
        typeDefinition.setAnnotatedDisplayName(displayName);
        typeDefinition.setDescription(description);
        typeDefinition.setRequired(required);
        typeDefinition.setDefaultValue(defaultValue);
        typeDefinition.setExample(example);
        return new Parameter(name, ParameterType.SECURITY, typeDefinition, isPassword);
    }

}
