package org.mule.connectivity.restconnect.internal.modelGeneration.amf.security;

import amf.client.model.StrField;
import amf.client.model.domain.*;
import edu.emory.mathcs.backport.java.util.Arrays;
import org.mule.connectivity.restconnect.exception.GenerationException;
import org.mule.connectivity.restconnect.exception.UnsupportedSecuritySchemeException;
import org.mule.connectivity.restconnect.internal.model.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.model.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.model.security.*;
import org.mule.connectivity.restconnect.internal.modelGeneration.JsonSchemaPool;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Collectors;

import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.util.AMFParserUtil.getParameterList;

public class AMFSecuritySchemeFactory {

    public static List<APISecurityScheme> getSchemesForOperation(EndPoint endPoint, Operation operation, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<SecurityScheme> globalSchemes = endPoint.security().stream().map(x -> getScheme(x)).collect(Collectors.toList());

        List<SecurityScheme> endPointSchemes = endPoint.security().stream().map(x -> getScheme(x)).collect(Collectors.toList());

        List<SecurityScheme> operationSchemes = operation.security().stream().map(x -> (SecurityScheme)x).collect(Collectors.toList());

        return getOperationSecuritySchemes(operationSchemes, endPointSchemes, globalSchemes, jsonSchemaPool);
    }

    private static List<APISecurityScheme> getOperationSecuritySchemes( List<SecurityScheme> operationSchemes, List<SecurityScheme> endPointSchemes, List<SecurityScheme> globalSchemes, JsonSchemaPool jsonSchemaPool) throws GenerationException {

        //If the method defines security, we validate it is supported and create the schemes
        if(!operationSchemes.isEmpty()){
            return getAPISecuritySchemes(operationSchemes, jsonSchemaPool);
        }

        //If the method does not defines security, we try with the endpoint
        else if(!endPointSchemes.isEmpty()){
            return getAPISecuritySchemes(endPointSchemes, jsonSchemaPool);
        }

        //If the endpoint does not defines security, we try with the global config
        else if(!globalSchemes.isEmpty()){
            return getAPISecuritySchemes(globalSchemes, jsonSchemaPool);
        }

        // If the method is not secured, then we generate the unsecured config.
        else{
            List<APISecurityScheme> returnSchemes = new LinkedList<>();
            returnSchemes.add(new UnsecuredScheme());
            return returnSchemes;
        }
    }

    private static List<APISecurityScheme> getAPISecuritySchemes(List<SecurityScheme> sourceSchemes, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<APISecurityScheme> returnSchemes = new LinkedList<>();

        for (SecurityScheme scheme : sourceSchemes) {
            List<APISecurityScheme> generatedSchemes = createSecuritySchemes(scheme, jsonSchemaPool);

            for(APISecurityScheme generatedScheme : generatedSchemes){
                if(returnSchemes.stream().noneMatch(x -> x.equals(generatedScheme))){
                    returnSchemes.add(generatedScheme);
                }
            }
        }

        // If the API is secured but we don't support any defined schemes, then we must throw a generation exception.
        if(!sourceSchemes.isEmpty() && returnSchemes.isEmpty()){
            throw new UnsupportedSecuritySchemeException("None of the specified security schemes ( " + listSchemes(sourceSchemes) + ") are supported.");
        }

        return returnSchemes;
    }

    private static List<APISecurityScheme> createSecuritySchemes(SecurityScheme scheme, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        ArrayList<APISecurityScheme> apiSecuritySchemes = new ArrayList<>();

        if (!scheme.type().nonEmpty()){
            apiSecuritySchemes.add(new UnsecuredScheme());
            return apiSecuritySchemes;
        }

        String schemeType = scheme.type().value();

        if(AMFSecuritySchemesNaming.isBasicAuth(schemeType)){
            apiSecuritySchemes.add(new BasicAuthScheme());
        }
        else if(AMFSecuritySchemesNaming.isPassThrough(schemeType)){
            apiSecuritySchemes.add(buildPassThroughSecurityScheme(scheme, jsonSchemaPool));
        }
        else if(AMFSecuritySchemesNaming.isOauth2(schemeType)){
            OAuth2Settings oAuth2Settings = (OAuth2Settings)scheme.settings();

            if (AMFOauth2FlowsNaming.isAuthorizationCode(oAuth2Settings)) {
                apiSecuritySchemes.add(buildOAuth2AuthorizationCodeSecurityScheme(oAuth2Settings));
            } else if (AMFOauth2FlowsNaming.isClientCredentials(oAuth2Settings)) {
                apiSecuritySchemes.add(buildOAuth2ClientCredentialsSecurityScheme(oAuth2Settings));
            }
        }
        else if(AMFSecuritySchemesNaming.isDigestAuth(schemeType)){
            apiSecuritySchemes.add(new DigestAuthenticationScheme());
        }
        else if(AMFSecuritySchemesNaming.isCustom(schemeType)){
            apiSecuritySchemes.add(buildCustomAuthenticationScheme(scheme, jsonSchemaPool));
        }

        return apiSecuritySchemes;
    }

    private static SecurityScheme getScheme(ParametrizedSecurityScheme securityScheme) {
        return (securityScheme.scheme().isLink() && securityScheme.scheme().linkTarget() != null) ? (SecurityScheme)securityScheme.scheme().linkTarget().get() : securityScheme.scheme();
    }

    private static PassThroughScheme buildPassThroughSecurityScheme(SecurityScheme securityScheme, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<Parameter> queryParameters = getParameterList(securityScheme.queryParameters(), ParameterType.SECURITY, jsonSchemaPool);
        List<Parameter> headers = getParameterList(securityScheme.headers(), ParameterType.SECURITY, jsonSchemaPool);

        return new PassThroughScheme(queryParameters, headers);
    }

    private static CustomAuthenticationScheme buildCustomAuthenticationScheme(SecurityScheme securityScheme, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<Parameter> queryParameters = getParameterList(securityScheme.queryParameters(), ParameterType.SECURITY, jsonSchemaPool);
        List<Parameter> headers = getParameterList(securityScheme.headers(), ParameterType.SECURITY, jsonSchemaPool);

        return new CustomAuthenticationScheme(queryParameters, headers);
    }

    private static OAuth2AuthorizationCodeScheme buildOAuth2AuthorizationCodeSecurityScheme(OAuth2Settings settings){
        String authorizationUri = settings.authorizationUri().value();
        String accessTokenUri = settings.accessTokenUri().value();
        List<String> authorizationGrants = Arrays.asList(new String[]{ OAuth2Scheme.OAUTH2_GRANT_AUTHORIZATION_CODE });
        List<String> scopes = settings.scopes().stream().map(s -> s.name().value()).collect(Collectors.toList());

        return new OAuth2AuthorizationCodeScheme(authorizationUri, accessTokenUri, authorizationGrants, scopes);
    }

    private static OAuth2ClientCredentialsScheme buildOAuth2ClientCredentialsSecurityScheme(OAuth2Settings settings){
        String authorizationUri = settings.authorizationUri().value();
        String accessTokenUri = settings.accessTokenUri().value();
        List<String> authorizationGrants = Arrays.asList(new String[]{ OAuth2Scheme.OAUTH2_GRANT_CLIENT_CREDENTIALS });
        List<String> scopes = settings.scopes().stream().map(s -> s.name().value()).collect(Collectors.toList());

        return new OAuth2ClientCredentialsScheme(authorizationUri, accessTokenUri, authorizationGrants, scopes);
    }

    private static String listSchemes(List<SecurityScheme> securitySchemes){
        StringBuilder builder = new StringBuilder();
        for(SecurityScheme securityScheme : securitySchemes){
            builder.append(securityScheme.name());
            builder.append(":");
            builder.append(securityScheme.type().value());
            if(securityScheme.type().value().equals(APISecurityScheme.OAUTH2) && securityScheme.settings() != null){
                OAuth2Settings settings = (OAuth2Settings)securityScheme.settings();
                builder.append(" :");
                if(settings.authorizationGrants() != null && settings.authorizationGrants().size() > 0){
                    for(StrField grant : settings.authorizationGrants()){
                        builder.append(" ");
                        builder.append(grant.value());
                    }
                }
            }
            builder.append(". ");
        }

        return builder.toString();
    }

}
