package org.mule.connectivity.restconnect.internal.model.operation;

import org.mule.connectivity.restconnect.exception.DuplicatedParameterNameException;
import org.mule.connectivity.restconnect.exception.InvalidSourceException;
import org.mule.connectivity.restconnect.internal.model.HTTPMethod;
import org.mule.connectivity.restconnect.internal.model.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.model.security.APISecurityScheme;
import org.mule.connectivity.restconnect.internal.model.type.TypeDefinition;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

public class OperationBuilder {

    private Operation operation;

    public OperationBuilder() {
        operation = new Operation();
    }

    public OperationBuilder withFriendlyName(String friendlyName){
        operation.setFriendlyName(friendlyName);
        return this;
    }

    public OperationBuilder withAnnotatedDisplayName(String annotatedDisplayName){
        operation.setAnnotatedDisplayName(annotatedDisplayName);
        return this;
    }

    public OperationBuilder withCanonicalName(String cannonicalName){
        operation.setCanonicalName(cannonicalName);
        return this;
    }

    public OperationBuilder withDescription(String description){
        operation.setDescription(description);
        return this;
    }

    public OperationBuilder withUri(String uri){
        operation.setUri(uri);
        return this;
    }

    public OperationBuilder withHttpMethod(HTTPMethod method){
        operation.setHttpMethod(method);
        return this;
    }

    public OperationBuilder withUriParams(List<Parameter> uriParams){
        operation.setUriParameters(uriParams);
        return this;
    }

    public OperationBuilder withQueryParams(List<Parameter> queryParams){
        operation.setQueryParameters(queryParams);
        return this;
    }

    public OperationBuilder withHeaders(List<Parameter> headers){
        operation.setHeaders(headers);
        return this;
    }

    public OperationBuilder withInputMetadata(TypeDefinition inputMetadata){
        operation.setInputMetadata(inputMetadata);
        return this;
    }

    public OperationBuilder withOutputMetadata(TypeDefinition outputMetadata){
        operation.setOutputMetadata(outputMetadata);
        return this;
    }

    public OperationBuilder withSecuritySchemes(List<APISecurityScheme> securitySchemes){
        operation.setSecuritySchemes(securitySchemes);
        return this;
    }

    private List<Parameter> buildParameters() throws InvalidSourceException {

        disambiguateAndValidateUniqueParameterNames(operation.getQueryParameters(), operation.getUriParameters(), operation.getHeaders());

        List<Parameter> allParams = new LinkedList<>();
        allParams.addAll(operation.getQueryParameters());
        allParams.addAll(operation.getUriParameters());
        allParams.addAll(operation.getHeaders());
        return allParams;
    }

    @SafeVarargs
    protected final void disambiguateAndValidateUniqueParameterNames(List<Parameter>... parameterSources) throws InvalidSourceException {
        disambiguateParameters(parameterSources);
        validateParameters(parameterSources);
    }

    private void validateParameters(List<Parameter>[] parameterSources) {
        List<String> keys = new ArrayList<>();

        for(List<Parameter> parameterSource : parameterSources){
            for(Parameter param : parameterSource){
                if(keys.stream().anyMatch(str -> str.equals(param.getInternalName()))){
                    throw new InvalidSourceException("Invalid spec: Duplicated parameter names", new DuplicatedParameterNameException(param.getExternalName(), operation.getCanonicalName(), operation.getUri()));
                }
                else {
                    keys.add(param.getInternalName());
                }
            }
        }
    }

    private void disambiguateParameters(List<Parameter>[] parameterSources) {
        // In order to mantain backwards compatibility we will only disambiguate parameters when there are several
        // declared using the same name and in the same operation. This is the case where we were failing to generate
        // the rest-connect model previously.
        List<Parameter> existingParameters = new ArrayList<>();

        List<Parameter> allParams = new LinkedList<>();
        for(List<Parameter> parameterSource : parameterSources){
            allParams.addAll(parameterSource);
        }

        for(Parameter parameter : allParams){
            for(Parameter existingParam : existingParameters){
                if(existingParam.getExternalName().equalsIgnoreCase(parameter.getExternalName())){
                    parameter.setDuplicatedName(true);
                    existingParam.setDuplicatedName(true);
                }
            }

            existingParameters.add(parameter);
        }
    }

    public Operation build() throws InvalidSourceException {
        operation.setParameters(buildParameters());
        return operation;
    }

}
