package org.mule.connectivity.restconnect.internal.modelGeneration.amf.util;

import amf.client.model.domain.*;
import org.mule.connectivity.restconnect.exception.GenerationException;
import org.mule.connectivity.restconnect.internal.model.HTTPMethod;
import org.mule.connectivity.restconnect.internal.model.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.model.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.modelGeneration.JsonSchemaPool;
import org.mule.connectivity.restconnect.internal.modelGeneration.util.ParserUtils;

import java.util.LinkedList;
import java.util.List;
import java.util.Optional;

import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.commons.lang3.StringUtils.trimToNull;
import static org.mule.connectivity.restconnect.internal.modelGeneration.amf.AMFTypeDefinitionFactory.getTypeDefinition;

public class AMFParserUtil extends ParserUtils{

    private static final String RC_PREFIX = "rest-connect.";
    private static final String RC_IGNORE_ANNOTATION = RC_PREFIX + IGNORE_ANNOTATION;
    private static final String RC_PARAMETER_NAME_ANNOTATION = RC_PREFIX + PARAMETER_NAME_ANNOTATION;
    private static final String RC_OPERATION_NAME_ANNOTATION = RC_PREFIX + OPERATION_NAME_ANNOTATION;

    public static String getMethodDisplayName(Operation operation) {
        String displayName = null;

        if(operation.name().nonEmpty() && !operation.name().value().equalsIgnoreCase(operation.method().value())){
            displayName = operation.name().value();
        }

        return trimToNull(displayName);
    }

    public static String getCanonicalOperationName(EndPoint endPoint, Operation operation, String friendlyName) {
        String annotatedOperationName = getAnnotatedOperationName(operation);

        HTTPMethod httpMethod = HTTPMethod.fromString(operation.method().value());

        return getCanonicalOperationName(httpMethod, endPoint.path().value(), friendlyName, annotatedOperationName);
    }

    public static String getAnnotatedParameterName(amf.client.model.domain.Parameter parameter) {
        String annotatedParameterName = getAnnotationForElement(parameter, RC_PARAMETER_NAME_ANNOTATION);

        if(isNotBlank(annotatedParameterName)){
            return annotatedParameterName;
        }

        return null;
    }

    public static String getAnnotatedOperationName(Operation operation) {
        String annotatedDisplayName = getAnnotationForElement(operation, RC_OPERATION_NAME_ANNOTATION);

        if(isNotBlank(annotatedDisplayName)){
            return annotatedDisplayName;
        }

        return null;
    }


    public static String getAnnotationForElement(DomainElement element, String annotationName){
        Optional<DomainExtension> optionalDomainExtension = element.customDomainProperties().stream()
                .filter(x -> x.name().nonEmpty() && annotationName.equalsIgnoreCase(x.name().value()))
                .findFirst();

        DomainExtension domainExtension;
        if(optionalDomainExtension.isPresent()){
            domainExtension = optionalDomainExtension.get();
        }
        else{
            return null;
        }

        return ((ScalarNode)domainExtension.extension()).value();
    }

    public static boolean elementContainsAnnotation(DomainElement element, String annotationName){
        return element.customDomainProperties().stream().anyMatch(x ->x.name().nonEmpty() && annotationName.equalsIgnoreCase(x.name().value()));
    }

    public static boolean isIgnored(EndPoint endPoint, Operation operation) {
        return elementContainsAnnotation(endPoint, RC_IGNORE_ANNOTATION)
                || elementContainsAnnotation(operation, RC_IGNORE_ANNOTATION);
    }

    public static List<Parameter> getParameterList(List<amf.client.model.domain.Parameter> amfParameters, ParameterType parameterType, JsonSchemaPool jsonSchemaPool) throws GenerationException {
        List<Parameter> parameters = new LinkedList<>();

        for (amf.client.model.domain.Parameter amfParameter : amfParameters) {
            parameters.add(new Parameter(amfParameter.name().value(), parameterType, getTypeDefinition(amfParameter, jsonSchemaPool)));
        }

        return parameters;
    }

    public static Shape getActualShape(Shape shape){
        if(shape == null)
            return null;

        if(shape.isLink() && shape.linkTarget().isPresent()){
            return getActualShape((Shape)shape.linkTarget().get());
        }
        return shape;
    }
}
