package org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.security;

import edu.emory.mathcs.backport.java.util.Arrays;
import org.mule.connectivity.restconnect.exception.UnsupportedSecuritySchemeException;
import org.mule.connectivity.restconnect.internal.model.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.model.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.model.security.*;
import org.mule.connectivity.restconnect.internal.modelGeneration.JsonSchemaPool;
import org.raml.v2.api.model.v10.api.Api;
import org.raml.v2.api.model.v10.methods.Method;
import org.raml.v2.api.model.v10.resources.Resource;
import org.raml.v2.api.model.v10.security.SecurityScheme;
import org.raml.v2.api.model.v10.security.SecuritySchemePart;

import java.util.LinkedList;
import java.util.List;
import java.util.stream.Collectors;

import static org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.util.RamlParserUtils.getParameterList;
import static org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.util.RamlParserUtils.getValueFromAnnotableString;


public class RamlParserSecuritySchemeFactory {


    public static List<APISecurityScheme> getOperationSecuritySchemes(Api api, Method method, JsonSchemaPool jsonSchemaPool) throws Exception {
        List<SecurityScheme> globalSchemes = api.securedBy().stream().map(x -> x != null ? x.securityScheme() : null).collect(Collectors.toList());

        Resource endPoint = method.resource();
        List<SecurityScheme> endPointSchemes = endPoint.securedBy().stream().map(x ->  x != null ? x.securityScheme() : null).collect(Collectors.toList());

        return getMethodSecuritySchemes(method, endPointSchemes, globalSchemes, jsonSchemaPool);
    }

    private static List<APISecurityScheme> getAPISecuritySchemes(List<SecurityScheme> sourceSchemes, JsonSchemaPool jsonSchemaPool) throws Exception {
        List<APISecurityScheme> returnSchemes = new LinkedList<>();

        for (SecurityScheme scheme : sourceSchemes) {
            List<APISecurityScheme> generatedSchemes = createSecuritySchemes(scheme, jsonSchemaPool);

            for(APISecurityScheme generatedScheme : generatedSchemes){
                if(returnSchemes.stream().noneMatch(x -> x.equals(generatedScheme))){
                    returnSchemes.add(generatedScheme);
                }
            }
        }

        // If the API is secured but we don't support any defined schemes, then we must throw a generation exception.
        if(!sourceSchemes.isEmpty() && returnSchemes.isEmpty()){
            throw new UnsupportedSecuritySchemeException("None of the specified security schemes ( " + listSchemes(sourceSchemes) + ") are supported.");
        }

        return returnSchemes;
    }

    private static List<APISecurityScheme> getMethodSecuritySchemes(Method method, List<SecurityScheme> endPointSchemes, List<SecurityScheme> globalSchemes, JsonSchemaPool jsonSchemaPool) throws Exception {
        List<SecurityScheme> methodSchemes = method.securedBy().stream().map(x -> x != null ? x.securityScheme() : null).collect(Collectors.toList());

        //If the method defines security, we validate it is supported and create the schemes
        if(!methodSchemes.isEmpty()){
            return getAPISecuritySchemes(methodSchemes, jsonSchemaPool);
        }

        //If the method does not defines security, we try with the endpoint
        else if(!endPointSchemes.isEmpty()){
            return getAPISecuritySchemes(endPointSchemes, jsonSchemaPool);
        }

        //If the endpoint does not defines security, we try with the global config
        else if(!globalSchemes.isEmpty()){
            return getAPISecuritySchemes(globalSchemes, jsonSchemaPool);
        }

        // If the method is not secured, then we generate the unsecured config.
        else{
            List<APISecurityScheme> returnSchemes = new LinkedList<>();
            returnSchemes.add(new UnsecuredScheme());
            return returnSchemes;
        }
    }


    private static List<APISecurityScheme> createSecuritySchemes(SecurityScheme securityScheme, JsonSchemaPool jsonSchemaPool) throws Exception {
        LinkedList<APISecurityScheme> apiSecuritySchemes = new LinkedList<>();

        if(securityScheme == null){
            apiSecuritySchemes.add(new UnsecuredScheme());
            return apiSecuritySchemes;
        }

        String schemeType = securityScheme.type();
        if(RamlParserSecuritySchemesNaming.isBasicAuth(schemeType)){
            apiSecuritySchemes.add(new BasicAuthScheme());
        }
        else if(RamlParserSecuritySchemesNaming.isPassThrough(schemeType)){
            apiSecuritySchemes.add(buildPassThroughSecurityScheme(securityScheme, jsonSchemaPool));
        }
        else if(RamlParserSecuritySchemesNaming.isOauth2(schemeType)){
            for (String grant: securityScheme.settings().authorizationGrants()) {

                if (RamlParserOauth2FlowsNaming.isAuthorizationCode(grant)) {
                    apiSecuritySchemes.add(buildOAuth2AuthorizationCodeSecurityScheme(securityScheme));
                } else if (RamlParserOauth2FlowsNaming.isClientCredentials(grant)) {
                    apiSecuritySchemes.add(buildOAuth2ClientCredentialsSecurityScheme(securityScheme));
                }
            }
        }
        else if(RamlParserSecuritySchemesNaming.isDigestAuth(schemeType)){
            apiSecuritySchemes.add(new DigestAuthenticationScheme());
        }
        else if(RamlParserSecuritySchemesNaming.isCustom(schemeType)){
            // If none of the previous types matches, it is a custom security scheme
            // https://github.com/raml-org/raml-spec/blob/master/versions/raml-10/raml-10.md#security-scheme-declaration
            apiSecuritySchemes.add(buildCustomAuthenticationSecurityScheme(securityScheme, jsonSchemaPool));
        }

        return apiSecuritySchemes;
    }

    private static PassThroughScheme buildPassThroughSecurityScheme(SecurityScheme securityScheme, JsonSchemaPool jsonSchemaPool) throws Exception {
        SecuritySchemePart describedBy = securityScheme.describedBy();
        List<Parameter> queryParameters = getParameterList(describedBy.queryParameters(), ParameterType.QUERY, jsonSchemaPool);
        List<Parameter> headers = getParameterList(describedBy.headers(), ParameterType.HEADER, jsonSchemaPool);

        return new PassThroughScheme(queryParameters, headers);
    }

    private static APISecurityScheme buildCustomAuthenticationSecurityScheme(SecurityScheme securityScheme, JsonSchemaPool jsonSchemaPool) throws Exception {
        SecuritySchemePart describedBy = securityScheme.describedBy();
        List<Parameter> queryParameters = getParameterList(describedBy.queryParameters(), ParameterType.QUERY, jsonSchemaPool);
        List<Parameter> headers = getParameterList(describedBy.headers(), ParameterType.HEADER, jsonSchemaPool);

        return new CustomAuthenticationScheme(queryParameters, headers);
    }

    private static OAuth2AuthorizationCodeScheme buildOAuth2AuthorizationCodeSecurityScheme(SecurityScheme securityScheme){
        String authorizationUri = getValueFromAnnotableString( securityScheme.settings().authorizationUri());
        String accessTokenUri = getValueFromAnnotableString( securityScheme.settings().accessTokenUri());
        List<String> authorizationGrants = Arrays.asList(new String[] { OAuth2Scheme.OAUTH2_GRANT_AUTHORIZATION_CODE });
        List<String> scopes = securityScheme.settings().scopes();

        return new OAuth2AuthorizationCodeScheme(authorizationUri, accessTokenUri, authorizationGrants, scopes);
    }

    private static OAuth2ClientCredentialsScheme buildOAuth2ClientCredentialsSecurityScheme(SecurityScheme securityScheme){
        String authorizationUri = getValueFromAnnotableString( securityScheme.settings().authorizationUri());
        String accessTokenUri = getValueFromAnnotableString( securityScheme.settings().accessTokenUri());
        List<String> authorizationGrants = Arrays.asList(new String[] { OAuth2Scheme.OAUTH2_GRANT_CLIENT_CREDENTIALS });
        List<String> scopes = securityScheme.settings().scopes();

        return new OAuth2ClientCredentialsScheme(authorizationUri, accessTokenUri, authorizationGrants, scopes);
    }

    private static String listSchemes(List<SecurityScheme> securitySchemes){
        StringBuilder builder = new StringBuilder();
        for(SecurityScheme securityScheme : securitySchemes){
            builder.append(securityScheme.type());
            builder.append("<");
            builder.append(securityScheme.name());
            builder.append(">");
            if(securityScheme.type().equals(APISecurityScheme.OAUTH2)){
                builder.append(" :");
                for(String grant : securityScheme.settings().authorizationGrants()){
                    builder.append(" ");
                    builder.append(grant);
                }
            }
            builder.append(". ");
        }

        return builder.toString();
    }

}
