package org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.util;

import org.mule.connectivity.restconnect.internal.model.HTTPMethod;
import org.mule.connectivity.restconnect.internal.model.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.model.parameter.ParameterType;
import org.mule.connectivity.restconnect.internal.modelGeneration.JsonSchemaPool;
import org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.predicate.OperationNamePredicate;
import org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.predicate.ParameterNamePredicate;
import org.mule.connectivity.restconnect.internal.modelGeneration.util.ParserUtils;
import org.raml.v2.api.model.v10.datamodel.FileTypeDeclaration;
import org.raml.v2.api.model.v10.datamodel.TypeDeclaration;
import org.raml.v2.api.model.v10.declarations.AnnotationRef;
import org.raml.v2.api.model.v10.methods.Method;
import org.raml.v2.api.model.v10.system.types.AnnotableStringType;

import javax.ws.rs.core.MediaType;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.StreamSupport;

import static com.google.common.collect.Iterables.filter;
import static org.apache.commons.lang3.StringUtils.trimToNull;
import static org.mule.connectivity.restconnect.internal.modelGeneration.ramlParser.RamlParserTypeDefinitionFactory.*;


public class RamlParserUtils extends ParserUtils {

    // Overrides default construction to avoiding instantiation.
    protected RamlParserUtils() {

    }

    public static String getMethodDescription(Method method) {
        String description = null;
        if(method.description() != null)
            description = method.description().value();

        else if(method.resource() != null && method.resource().description() != null)
            description = method.resource().description().value();

        return trimToNull(description);
    }

    public static String getMethodDisplayName(Method method) {
        String displayName = null;
        // This checks for the case that the displayName attribute isn't provided, as the parsers
        // defaults this to the method name, per the RAML 1.0 spec.
        if(method.displayName() != null && method.displayName().value() != null && !method.displayName().value().equals(method.method()))
            displayName = method.displayName().value();

        return trimToNull(displayName);
    }


    public static String getCanonicalOperationName(Method method, String friendlyName) {
        String annotatedOperationName = getAnnotatedOperationName(method);

        HTTPMethod httpMethod = HTTPMethod.fromString(method.method());

        return getCanonicalOperationName(httpMethod, method.resource().resourcePath(), friendlyName, annotatedOperationName);
    }

    public static String getAnnotatedOperationName(Method method) {
        List<AnnotationRef> annotations = method.annotations();
        if(annotations != null && !annotations.isEmpty()){
            AnnotationRef opNameAnnotation = StreamSupport.stream(filter(annotations, new OperationNamePredicate()).spliterator(), false).findFirst().orElse(null);
            if(opNameAnnotation != null){
                return (String) opNameAnnotation.structuredValue().value();
            }
        }

        return null;
    }

    public static String getAnnotatedParameterName(TypeDeclaration typeDeclaration) {
        List<AnnotationRef> annotations = typeDeclaration.annotations();
        if(annotations != null && !annotations.isEmpty()){
            AnnotationRef parameterNameAnnotation = StreamSupport.stream(filter(annotations, new ParameterNamePredicate()).spliterator(), false).findFirst().orElse(null);
            if(parameterNameAnnotation != null){
                return (String) parameterNameAnnotation.structuredValue().value();
            }
        }

        return null;
    }

    public static List<Parameter> getParameterList(List<TypeDeclaration> typeDeclarationParameters, ParameterType parameterType, JsonSchemaPool jsonSchemaPool) throws Exception {
        return getParameterList(typeDeclarationParameters, false, parameterType, jsonSchemaPool);
    }

    public static List<Parameter> getParameterList(List<TypeDeclaration> typeDeclarationParameters, boolean setDefaultMediaTypes, ParameterType parameterType, JsonSchemaPool jsonSchemaPool) throws Exception {
        List<Parameter> parameters = new LinkedList<>();

        for (TypeDeclaration typeDeclaration : typeDeclarationParameters) {
            if(setDefaultMediaTypes){
                parameters.add(new Parameter(typeDeclaration.name(), parameterType, getTypeDefinition(typeDeclaration, getDefaultMediaType(typeDeclaration), jsonSchemaPool)));
            }
            else{
                parameters.add(new Parameter(typeDeclaration.name(), parameterType, getTypeDefinition(typeDeclaration, jsonSchemaPool)));
            }
        }

        return parameters;
    }

    private static String getDefaultMediaType(TypeDeclaration typeDeclaration) {
        if(typeDeclaration instanceof FileTypeDeclaration)
            return MediaType.APPLICATION_OCTET_STREAM;

        if(typeIsDefinedWithRAML(typeDeclaration))
            return MediaType.APPLICATION_JSON;

        if(typeIsDefinedWithJSONSchema(typeDeclaration))
            return MediaType.APPLICATION_JSON;

        if(typeIsDefinedWithXMLSchema(typeDeclaration))
            return MediaType.APPLICATION_XML;

        return MediaType.TEXT_PLAIN;
    }

    public static String getValueFromAnnotableString(AnnotableStringType annotableString) {
        return annotableString == null ? null : annotableString.value();
    }

    public static boolean isIgnored(Method method) {
        for (AnnotationRef annotationRef : method.annotations()) {
            if(IGNORE_ANNOTATION.equals(annotationRef.annotation().name()))
                return true;
        }

        return false;
    }
}
