package org.mule.connectivity.restconnect.internal.templateEngine;

import org.apache.velocity.VelocityContext;
import org.apache.velocity.tools.ToolManager;
import org.mule.connectivity.restconnect.internal.model.security.APISecurityScheme;
import org.mule.connectivity.restconnect.internal.model.security.OAuth2Scheme;
import org.mule.connectivity.restconnect.internal.templateEngine.builder.SmartConnectorTemplateEngineBuilder;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.model.SmartConnectorModelDecorator;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.operation.SmartConnectorOperationDecorator;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.type.SmartConnectorTypeDefinitionDecorator;
import org.mule.connectivity.restconnect.internal.util.FileGenerationUtils;
import org.w3c.dom.Document;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXParseException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import java.io.*;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Properties;

import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.join;


public class SmartConnectorTemplateEngine extends TemplateEngine {

    private static final String EXTENSION = "extension";

    private static final String CATALOG_VM = "templates/smartConnector/catalog.vm";
    private static final String MODULE_VM = "templates/smartConnector/module.vm";
    private static final String MULE_ARTIFACT_VM = "templates/smartConnector/mule-artifact.vm";
    private static final String POM_VM = "templates/smartConnector/pom.vm";
    private static final String ICON_PATH = "jarResources/images/icon.svg";
    private static final Path RESOURCES_DIR = Paths.get("src/main/resources");

    private static final String PROPERTIES_RESOURCE = "/project.properties";
    private static final String APPLICATION_PROPERTIES = "properties";

    private final SmartConnectorModelDecorator model;
    private final Path outputDir;
    private SmartConnectorGeneratedSources generatedSources;


    public SmartConnectorTemplateEngine(SmartConnectorTemplateEngineBuilder builder) {
        this.model = new SmartConnectorModelDecorator(builder.getModel());
        this.outputDir = builder.getOutputDir();
        this.generatedSources = new SmartConnectorGeneratedSources(outputDir);
    }

    @Override
    public void applyTemplates() throws Exception {
        generateCatalog();
        generateVMTemplates();
        generateAssetFiles();
    }

    private void generateAssetFiles() throws IOException {
        generateIconFile();
    }

    private void generateIconFile() throws IOException {
        ClassLoader classLoader = getClass().getClassLoader();

        InputStream stream = null;
        OutputStream outStream = null;

        try{
            Path output = outputDir.resolve("icon.svg");

            stream = classLoader.getResourceAsStream(ICON_PATH);
            byte[] buffer = new byte[stream.available()];
            stream.read(buffer);


            File targetFile = output.toFile();
            outStream = new FileOutputStream(targetFile);
            outStream.write(buffer);

            generatedSources.setIcon(output);
        }
        finally {
            if(stream != null){
                stream.close();
            }
            if(outStream != null){
                outStream.close();
            }
        }
    }

    private void generateCatalog() throws IOException {
        Path outputResourcesDir = outputDir.resolve(RESOURCES_DIR);
        for (SmartConnectorOperationDecorator operation : model.getDecoratedOperations()) {
            if(operation.getDecoratedInputMetadata() != null && operation.getDecoratedInputMetadata().requiresCatalog()) {
                generatedSources.addCatalogSchema(operation.getDecoratedInputMetadata().writeSchema(outputResourcesDir));
            }

            if(operation.getDecoratedOutputMetadata() != null && operation.getDecoratedOutputMetadata().requiresCatalog()) {
                generatedSources.addCatalogSchema(operation.getDecoratedOutputMetadata().writeSchema(outputResourcesDir));
            }

            for(SmartConnectorTypeDefinitionDecorator parameter : operation.getDecoratedParameters()) {
                if(parameter.requiresCatalog()) {
                    generatedSources.addCatalogSchema(parameter.writeSchema(outputResourcesDir));
                }
            }
        }

        addXmlSchemasRecursivelyToOutput(model.getRootDir().toFile(), outputResourcesDir);
    }

    public void addXmlSchemasRecursivelyToOutput(File file, Path outputDirectory){
        if(!file.exists()){
            return;
        }

        if(file.isDirectory() && !file.getName().equalsIgnoreCase("target")){
            for(File subFile : Objects.requireNonNull(file.listFiles())){
                addXmlSchemasRecursivelyToOutput(subFile, outputDirectory);
            }
        }

        try{
            DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance();
            DocumentBuilder dBuilder = dbFactory.newDocumentBuilder();
            dBuilder.setErrorHandler(new ErrorHandler() {
                @Override
                public void warning(SAXParseException exception) {}
                @Override
                public void error(SAXParseException exception) {}
                @Override
                public void fatalError(SAXParseException exception) {}
            });
            Document doc = dBuilder.parse(file);

            copyRelativizedFile(outputDirectory, file.toPath());
            generatedSources.addComplementaryXmlSchema(file.toPath());
        }
        catch(Exception e){
            return;
        }
    }

    private void generateVMTemplates() throws Exception {
        ToolManager velocityToolManager = new ToolManager();
        velocityToolManager.configure("velocity-tools.xml");
        VelocityContext context = new VelocityContext(velocityToolManager.createContext());

        context.internalPut(EXTENSION, model);
        context.internalPut(APPLICATION_PROPERTIES, getApplicationProperties());

        Path outputResourcesDir = outputDir.resolve(RESOURCES_DIR);

        generatedSources.setPom(applyTemplate(POM_VM, outputDir.resolve("pom.xml"), context));
        generatedSources.setModule(applyTemplate(MODULE_VM, outputResourcesDir.resolve(format("module-%s.xml", model.getModulePrefix())), context));
        generatedSources.setCatalog(applyTemplate(CATALOG_VM, outputResourcesDir.resolve(format("module-%s-catalog.xml", model.getModulePrefix())), context));
        generatedSources.setArtifact(applyTemplate(MULE_ARTIFACT_VM, outputDir.resolve("mule-artifact.json"), context));
    }

    public void packageProject() throws Exception {
        Path packageFolder = outputDir.resolve("package");
        Files.createDirectories(packageFolder);
        createJarFile(packageFolder);
    }

    private void createJarFile(Path packageFolder) throws Exception {
        Path jarDir = Files.createTempDirectory("jar-temp");

        Path metaInfDir = jarDir.resolve(Paths.get("META-INF"));
        metaInfDir.toFile().mkdirs();
        Path muleArtifactDir = metaInfDir.resolve(Paths.get("mule-artifact"));
        muleArtifactDir.toFile().mkdirs();
        Path mavenDir = metaInfDir.resolve(Paths.get("maven/" + model.getGroupId() + "/" + model.getArtifactId()));
        mavenDir.toFile().mkdirs();

        Files.copy(generatedSources.getCatalog(), jarDir.resolve(generatedSources.getCatalog().getFileName()));
        Files.copy(generatedSources.getModule(), jarDir.resolve(generatedSources.getModule().getFileName()));
        Files.copy(generatedSources.getArtifact(), muleArtifactDir.resolve(generatedSources.getArtifact().getFileName()));
        Files.copy(generatedSources.getIcon(), muleArtifactDir.resolve(generatedSources.getIcon().getFileName()));
        Files.copy(generatedSources.getPom(), mavenDir.resolve(generatedSources.getPom().getFileName()));

        for (Path path : generatedSources.getDistinctCatalogSchemas()){
            Files.copy(path, jarDir.resolve(path.getFileName()));
        }

        for (Path path : generatedSources.getDistinctComplementaryXmlSchemas()){
            copyRelativizedFile(jarDir, path);
        }

        String jarName = format("%s-%s-mule-plugin.jar", model.getArtifactId(), model.getVersion());
        Path outputJarFile = packageFolder.resolve(jarName);
        FileGenerationUtils.generateJarFileFromDirectory(jarDir, outputJarFile);
    }

    private void copyRelativizedFile(Path destinationDir, Path fileToCopy) throws IOException {
        String stringPathInJar = fileToCopy.toString().replace(model.getRootDir().toString() + "/", "");
        stringPathInJar = stringPathInJar.replace(model.getRootDir().toString() + "\\", "");
        stringPathInJar = stringPathInJar.replace(model.getRootDir().toString(), "");

        Path jarPath = destinationDir.resolve(stringPathInJar);
        if(!jarPath.getParent().toFile().exists()){
            jarPath.getParent().toFile().mkdirs();
        }
        Files.copy(fileToCopy, jarPath);
    }

    private Properties getApplicationProperties() throws IOException {
        try(InputStream input = this.getClass().getResourceAsStream(PROPERTIES_RESOURCE)) {
            Properties prop = new Properties();
            prop.load(input);
            return prop;
        }
    }

    private String getGrants(List<APISecurityScheme> schemas){
        List<String> grants = new ArrayList<>();

        for(APISecurityScheme schema : schemas){
            if(schema instanceof OAuth2Scheme){
                grants.addAll(((OAuth2Scheme) schema).getAuthorizationGrants());
            }
        }

        return join(grants, ", ");
    }

}
