package org.mule.connectivity.restconnect.internal.templateEngine.decorator.model;

import com.google.common.collect.ImmutableList;
import org.apache.commons.lang3.StringUtils;
import org.mule.connectivity.restconnect.internal.model.api.RestConnectAPIModel;
import org.mule.connectivity.restconnect.internal.model.operation.Operation;
import org.mule.connectivity.restconnect.internal.model.security.APISecurityScheme;
import org.mule.connectivity.restconnect.internal.templateEngine.PojoFQNamePool;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.operation.DevKitOperationDecorator;
import org.mule.connectivity.restconnect.internal.templateEngine.decorator.security.devkit.DevKitSecuritySchemeDecorator;

import java.nio.file.Path;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.connectivity.restconnect.internal.modelGeneration.util.ParserUtils.*;


public class DevKitConnectorModelDecorator extends ModelDecorator {

    public static final String DEFAULT_ARTIFACT_ID_PREFIX = "mule-module-";
    public static final String DEFAULT_GROUP_ID = "org.mule.modules";

    private final List<DevKitOperationDecorator> decoratedOperations;
    private final String basePackage;
    private final Map<APISecurityScheme, String> securitySchemeNames;
    private List<DevKitSecuritySchemeDecorator> securitySchemes;
    private final PojoFQNamePool pojoFQNamePool;

    public DevKitConnectorModelDecorator(RestConnectAPIModel model) {
        super(model);
        this.pojoFQNamePool = new PojoFQNamePool();
        this.securitySchemeNames = new HashMap<>();
        this.securitySchemes = new LinkedList<>();
        this.basePackage = createBasePackage();
        this.decoratedOperations = buildDecoratedOperations();
    }

    private List<DevKitOperationDecorator> buildDecoratedOperations() {
        ImmutableList.Builder<DevKitOperationDecorator> builder = ImmutableList.builder();

        for (Operation operation : this.getOperations()) {
            builder.add(new DevKitOperationDecorator(operation, basePackage, securitySchemeNames, pojoFQNamePool));
        }

        return builder.build();
    }

    private String createBasePackage(){
        if(super.getCustomPackage() != null){
            return removeJavaPackageUnwantedCharacters(getCustomPackage()) + "." + removeJavaNameUnwantedCharacters(getApiName()).toLowerCase();
        }
        else{
            return removeJavaPackageUnwantedCharacters(getGroupId()) + "." + removeJavaNameUnwantedCharacters(getApiName()).toLowerCase();
        }
    }

    @Override
    public String getGroupId() {
        return isNotBlank(super.getGroupId()) ? super.getGroupId() : DEFAULT_GROUP_ID;
    }

    @Override
    public String getArtifactId() {
        return isNotBlank(super.getArtifactId()) ? super.getArtifactId() : getDefaultArtifactId();
    }

    private String getDefaultArtifactId() {
        return DEFAULT_ARTIFACT_ID_PREFIX + removeSpaces(getApiName()).toLowerCase();
    }

    public String getConnectorName() {
        String xmlValid = getXmlName(getApiName());
        if(xmlValid.toLowerCase().startsWith("_xml") || xmlValid.toLowerCase().startsWith("xml") || xmlValid.toLowerCase().startsWith("_")){
            return "connector" + xmlValid;
        }
        return xmlValid;
    }

    public String getBasePackage() {
        return this.basePackage;
    }

    public String getClassName() {
        return removeJavaNameUnwantedCharacters(getApiName()) + "Connector";
    }

    public List<DevKitOperationDecorator> getDecoratedOperations() {
        return this.decoratedOperations;
    }

    public String getBaseUriAsString() {
        return getBaseUri().toString();
    }

    public Path getRootDir() {
        return model.getRootDir();
    }

    public String getApiDescription() {
        if(StringUtils.isNotBlank(model.getApiDescription())){
            String lineSeparator = System.getProperty("line.separator");

            StringBuilder builder = new StringBuilder();
            builder.append("/**").append(lineSeparator)
                   .append("*").append(model.getApiDescription().replace(lineSeparator, lineSeparator + "*")).append(lineSeparator)
                   .append("*/").append(lineSeparator);

            return builder.toString();
        }
        else{
            return "";
        }
    }

    public List<DevKitSecuritySchemeDecorator> getSecuritySchemes() {
        if(securitySchemes.isEmpty()){
            securitySchemes = getOperationsSecuritySchemes();
        }
        return securitySchemes;
    }

    private List<DevKitSecuritySchemeDecorator> getOperationsSecuritySchemes(){
        List<DevKitSecuritySchemeDecorator> operationSchemes = new LinkedList<>();

        for(DevKitOperationDecorator operation : getDecoratedOperations()){

            for(DevKitSecuritySchemeDecorator scheme : operation.getSecuritySchemes()){
                if(operationSchemes.stream().noneMatch(x -> x.getApiSecurityScheme().equals(scheme.getApiSecurityScheme()))){
                    operationSchemes.add(scheme);
                }
            }
        }

        return operationSchemes;
    }

}
