package org.mule.connectivity.restconnect.internal.templateEngine.decorator.type;

import org.mule.connectivity.restconnect.exception.GenerationException;
import org.mule.connectivity.restconnect.internal.model.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.model.typesource.PrimitiveTypeSource;
import org.mule.connectivity.restconnect.internal.templateEngine.PojoFQNamePool;
import org.mule.connectivity.restconnect.internal.util.FileGenerationUtils;

import java.io.File;
import java.nio.file.Path;

import static org.mule.connectivity.restconnect.internal.modelGeneration.util.ParserUtils.removeJavaNameUnwantedCharacters;

public class DevKitConnectorTypeDefinitionDecorator extends TypeDefinitionDecorator {

    private final String basePackage;
    private String fullQualifiedName;
    private Path pojoLocation;
    private String className;
    private boolean failedToGeneratePojo = false;
    private final PojoFQNamePool pojoFQNamePool;

    public DevKitConnectorTypeDefinitionDecorator(String externalName, String internalName, TypeDefinition typeDefinition, String basePackage, PojoFQNamePool pojoFQNamePool) {
        super(externalName, internalName, typeDefinition);
        this.basePackage = basePackage;
        this.pojoFQNamePool = pojoFQNamePool;
    }

    public DevKitConnectorTypeDefinitionDecorator(TypeDefinition inputMetadata, String basePackage, PojoFQNamePool pojoFQNamePool) {
        super(inputMetadata);
        this.basePackage = basePackage;
        this.pojoFQNamePool = pojoFQNamePool;
    }

    public void generatePojo(Path outputDir) throws GenerationException {
        if(pojoFQNamePool.containsTypeDefinition(getTypeDefinition())){
            fullQualifiedName = pojoFQNamePool.getFullyQualifiedName(getTypeDefinition());
        }
        else{
            fullQualifiedName = FileGenerationUtils.generatePojo(getTypeDefinition().getSource(), getClassName(), outputDir.toFile(), basePackage);
            pojoFQNamePool.putTypeDefinitionFQNamePair(getTypeDefinition(), fullQualifiedName);
        }

        pojoLocation = outputDir.resolve((fullQualifiedName.replace(".", File.separator)) + ".java").toAbsolutePath();
    }

    public Path getPojoLocation(){
        return pojoLocation;
    }

    public void setClassName(String className) {
        this.className = className;
    }

    public String getClassName() {
        //Forced to String for union types
        //Forced to string when the POJO can not be generated
        if(isUnionType() || failedToGeneratePojo){
            return "String";
        }

        if(isPrimitiveType()){
            return getJavaPrimitiveType();
        }

        if(fullQualifiedName != null){
            return fullQualifiedName;
        }

        return className;
    }

    public String getRequestBuilderResponseType(){
        return getJavaGenericArgument(getClassName());
    }

    private String getJavaGenericArgument(String className) {
        if (className.contains("java.util.List<")) {
            String strippedClassName = className
                    .replaceFirst("java.util.List<", "")
                    .replaceFirst(">", "");
            return getJavaGenericArgument("java.util.List.class, " + strippedClassName);
        }
        return className + ".class";
    }

    private String getJavaPrimitiveType() {
        PrimitiveTypeSource typeSource = (PrimitiveTypeSource) getTypeDefinition().getSource();
        switch (typeSource.getType()) {
            case BOOLEAN:
                return "Boolean";
            case NUMBER:
            case INTEGER:
                return "Double";
            case DATE:
            case DATE_TIME:
            case DATE_ONLY:
            case DATE_TIME_ONLY:
            case TIME_ONLY:
                return "Date";
            case FILE:
                return "File";
            case STRING:
                return "String";
        }

        throw new IllegalArgumentException("Invalid type value: " + typeSource.getType());
    }

    @Override
    public String getExternalName() {
        return super.getExternalName();
    }

    @Override
    public String getInternalName() {
        return removeJavaNameUnwantedCharacters(super.getInternalName());
    }

    public boolean requiresPojo() {
        return ( isEnumType() ) || ( !isPrimitiveType() && !isUnionType() ) && !failedToGeneratePojo;
    }

    public void setFailedToGeneratePojo(boolean failed){
        failedToGeneratePojo = failed;
    }

    public boolean isFailedToGeneratePojo(){
        return failedToGeneratePojo;
    }
}
