/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.templating;

import static com.google.common.base.CaseFormat.UPPER_CAMEL;
import org.mule.connectivity.restconnect.exception.TemplatingException;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorModel;

import com.google.common.base.CaseFormat;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeSpec;

import java.nio.file.Path;
import java.util.Optional;

import javax.lang.model.element.Modifier;


public abstract class JavaTemplateEntity extends TemplateEntity {

  protected static final String VALUE_MEMBER = "value";
  protected static final String NAME_MEMBER = "name";

  protected final Path outputDir;
  protected final ConnectorModel connectorModel;

  public JavaTemplateEntity(Path outputDir, ConnectorModel connectorModel) {
    this.outputDir = outputDir;
    this.connectorModel = connectorModel;
  }

  protected void writeClassToFile(TypeSpec clazz, String targetPackage) throws TemplatingException {
    writeClassToFile(JavaFile.builder(targetPackage, clazz).skipJavaLangImports(true).build());
  }

  protected void writeClassToFile(JavaFile javaFile) throws TemplatingException {
    try {
      javaFile.writeTo(outputDir.resolve("src/main/java"));
    } catch (Exception e) {
      throw new TemplatingException("There was an error when writing " + this.getClass().getName() + " class", e);
    }
  }

  public Path getResourcesPath() {
    return outputDir.resolve("src/main/resources");
  }

  public Path getSourcesPath() {
    return outputDir.resolve("src");
  }

  protected MethodSpec.Builder generateGetter(FieldSpec fieldSpec, CaseFormat fieldNameCaseFormat) {
    String name = "get" + fieldNameCaseFormat.to(UPPER_CAMEL, fieldSpec.name);

    return MethodSpec
        .methodBuilder(name)
        .addModifiers(Modifier.PUBLIC)
        .returns(fieldSpec.type)
        .addCode(CodeBlock.builder().addStatement("return this.$N", fieldSpec).build());
  }

  protected MethodSpec.Builder generateOptionalGetter(FieldSpec fieldSpec, Class<?> type, CaseFormat fieldNameCaseFormat) {
    String name = "get" + fieldNameCaseFormat.to(UPPER_CAMEL, fieldSpec.name);

    return MethodSpec
        .methodBuilder(name)
        .addModifiers(Modifier.PUBLIC)
        .returns(ParameterizedTypeName.get(Optional.class, type))
        .addCode(CodeBlock.builder().addStatement("return $T.ofNullable(this.$N)", Optional.class, fieldSpec).build());
  }
}
