/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.validation.rules;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.connectivity.restconnect.internal.validation.ValidationRule.Level.ERROR;
import org.mule.connectivity.restconnect.internal.descriptor.model.ConnectorDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.EndPointDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.OperationDescriptor;
import org.mule.connectivity.restconnect.internal.validation.PreValidationRule;
import org.mule.connectivity.restconnect.internal.validation.ValidationResult;
import org.mule.connectivity.restconnect.internal.webapi.model.APIModel;

import java.util.LinkedList;
import java.util.List;

public class DescriptorRedundantOutputDefinitionRule extends PreValidationRule {

  //R021
  public DescriptorRedundantOutputDefinitionRule() {
    super("Operation descriptor's properties skipOutputTypeValidation, voidOperation and outputType are mutually exclusive.",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationResult> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {

    List<ValidationResult> allValidationResults = new LinkedList<>();

    for (EndPointDescriptor endpoint : connectorDescriptor.getEndpoints()) {
      List<ValidationResult> validationResults = endpoint.getOperations().stream()
          .filter(this::moreThanOneExclusiveProperty)
          .map(operation -> getValidationError(endpoint.getPath(), operation.getMethod()))
          .collect(toList());

      allValidationResults.addAll(validationResults);
    }

    return allValidationResults;
  }

  private boolean moreThanOneExclusiveProperty(OperationDescriptor operationDescriptor) {
    int count = 0;

    if (operationDescriptor.getVoidOperation() != null && operationDescriptor.getVoidOperation()) {
      count++;
    }

    if (operationDescriptor.getSkipOutputTypeValidation() != null && operationDescriptor.getSkipOutputTypeValidation()) {
      count++;
    }

    if (isNotBlank(operationDescriptor.getOutputTypeSchema())) {
      count++;
    }

    return count > 1;
  }

  private ValidationResult getValidationError(String path, String method) {
    String location =
        "Operation with PATH: " + path +
            " and METHOD: " + method + " defined in the Connector Descriptor";
    return new ValidationResult(this, location);
  }
}
