/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.connectormodel.type;

import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.PartParameter;
import org.mule.connectivity.restconnect.internal.connectormodel.type.schema.TypeSchema;
import org.mule.connectivity.restconnect.internal.util.HashCodeUtil;

import javax.ws.rs.core.MediaType;
import java.util.List;

public abstract class TypeDefinition {


  protected MediaType mediaType;
  protected String example;
  protected List<String> enumValues;
  protected TypeSchema typeSchema;

  TypeDefinition(MediaType mediaType,
                 String example,
                 List<String> enumValues,
                 TypeSchema typeSchema) {

    this.mediaType = mediaType;
    this.example = example;
    this.enumValues = enumValues;
    this.typeSchema = typeSchema;
  }

  public MediaType getMediaType() {
    return mediaType;
  }

  public String getExample() {
    return example;
  }

  public List<String> getEnumValues() {
    return enumValues;
  }

  public TypeSchema getTypeSchema() {
    return typeSchema;
  }

  public boolean isEnum() {
    return enumValues != null && !enumValues.isEmpty();
  }

  public static TypeDefinition simplePrimitiveType(PrimitiveTypeDefinition.PrimitiveType type) {
    return new PrimitiveTypeDefinition(MediaType.TEXT_PLAIN_TYPE, null, null, null, type);
  }

  public static TypeDefinition simpleStringType() {
    return simplePrimitiveType(PrimitiveTypeDefinition.PrimitiveType.STRING);
  }

  @Override
  public int hashCode() {
    String mediaTypeString = this.mediaType != null ? this.mediaType.toString() : null;
    return HashCodeUtil.generateHashCode(mediaTypeString, example, enumValues, typeSchema);
  }

  public static Builder builder(MediaType mediaType, TypeSchema typeSchema, List<String> enumValues, String example)
      throws ModelGenerationException {
    if (mediaType == null) {
      throw new ModelGenerationException("Media Type can not be empty");
    }
    return new Builder(mediaType, typeSchema, enumValues, example);
  }

  public static class Builder {

    private final MediaType mediaType;
    private final TypeSchema typeSchema;
    private final List<String> enumValues;
    private final String example;

    Builder(MediaType mediaType, TypeSchema typeSchema, List<String> enumValues, String example) {
      this.mediaType = mediaType;
      this.typeSchema = typeSchema;
      this.enumValues = enumValues;
      this.example = example;
    }

    public ArrayTypeDefinition buildArrayType(TypeDefinition innerType) {
      return new ArrayTypeDefinition(mediaType, example, enumValues, typeSchema, innerType);
    }

    public MultipartTypeDefinition buildMultipartTypeDefinition(List<PartParameter> parts) {
      return new MultipartTypeDefinition(example, enumValues, typeSchema, parts);
    }

    public ObjectTypeDefinition buildObjectTypeDefinition() {
      return new ObjectTypeDefinition(mediaType, example, enumValues, typeSchema);
    }

    public PrimitiveTypeDefinition buildPrimitiveTypeDefinition(PrimitiveTypeDefinition.PrimitiveType primitiveType) {
      return new PrimitiveTypeDefinition(mediaType, example, enumValues, typeSchema, primitiveType);
    }

    public UnionTypeDefinition buildUnionTypeDefinition(List<TypeDefinition> unionTypes) {
      return new UnionTypeDefinition(mediaType, example, enumValues, typeSchema, unionTypes);
    }

    public EmptyTypeDefinition buildEmptyTypeDefinition() {
      return new EmptyTypeDefinition(mediaType, example, enumValues);
    }
  }
}
