/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.descriptor.parser;

import static java.lang.Boolean.parseBoolean;
import static org.mule.connectivity.restconnect.internal.descriptor.parser.DescriptorParserUtils.getSchemaContent;
import static org.mule.connectivity.restconnect.internal.descriptor.parser.DescriptorParserUtils.parseElementCoreDescription;
import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.internal.descriptor.model.PartDescriptor;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

import amf.client.model.domain.DialectDomainElement;

public class DescriptorPartParser {

  private static final String API_CONTRACT_PART_NAME = "http://a.ml/vocabularies.restc/apiContract#paramName";
  private static final String REST_CONNECT_FRIENDLY_NAME = "http://a.ml/vocabularies/restConnect#friendlyName";
  private static final String REST_CONNECT_INPUT_TYPE = "http://a.ml/vocabularies/restConnect#inputType";
  private static final String REST_CONNECT_INPUT_MEDIA_TYPE = "http://a.ml/vocabularies/restConnect#inputMediaType";
  private static final String REST_CONNECT_FILE_PART = "http://a.ml/vocabularies/restConnect#filePart";


  public List<PartDescriptor> parseParts(List<DialectDomainElement> parameterElements, Path descriptorFilePath)
      throws ModelGenerationException {

    List<PartDescriptor> parts = new ArrayList<>();
    for (DialectDomainElement p : parameterElements) {
      parts.add(parsePart(p, descriptorFilePath));
    }

    return parts;
  }

  private PartDescriptor parsePart(DialectDomainElement partElement, Path descriptorFilePath)
      throws ModelGenerationException {
    return new PartDescriptor(getPartName(partElement),
                              getFriendlyName(partElement),
                              getDescription(partElement),
                              getInputTypeSchema(partElement, descriptorFilePath),
                              getContentType(partElement),
                              getFilePart(partElement));
  }

  private String getPartName(DialectDomainElement partElement) {
    List<Object> partName = partElement.getScalarByPropertyUri(API_CONTRACT_PART_NAME);
    return partName.isEmpty() ? null : partName.get(0).toString();
  }

  private String getFriendlyName(DialectDomainElement partElement) {
    List<Object> friendlyName = partElement.getScalarByPropertyUri(REST_CONNECT_FRIENDLY_NAME);
    return friendlyName.isEmpty() ? null : friendlyName.get(0).toString();
  }

  private String getDescription(DialectDomainElement partElement) {
    return parseElementCoreDescription(partElement);
  }

  private String getInputTypeSchema(DialectDomainElement partElement, Path descriptorDirectory)
      throws ModelGenerationException {
    List<Object> inputTypeSchema = partElement.getScalarByPropertyUri(REST_CONNECT_INPUT_TYPE);
    String inputSchemaPathString = inputTypeSchema.isEmpty() ? null : inputTypeSchema.get(0).toString();

    return getSchemaContent(descriptorDirectory, inputSchemaPathString);
  }

  private String getContentType(DialectDomainElement partElement) {
    List<Object> contentType = partElement.getScalarByPropertyUri(REST_CONNECT_INPUT_MEDIA_TYPE);
    return contentType.isEmpty() ? null : contentType.get(0).toString();
  }

  private boolean getFilePart(DialectDomainElement partElement) {
    List<Object> isFile = partElement.getScalarByPropertyUri(REST_CONNECT_FILE_PART);
    return !isFile.isEmpty() && parseBoolean(isFile.get(0).toString());
  }
}
