/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.util;

import static com.google.common.base.Charsets.UTF_8;
import static java.lang.String.format;
import static java.lang.String.join;
import static java.util.regex.Pattern.compile;
import static org.apache.commons.io.FileUtils.writeStringToFile;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.connectivity.restconnect.internal.util.FileGenerationUtils.SchemaNameType.INPUT;
import static org.mule.connectivity.restconnect.internal.webapi.util.OperationNamingUtils.getPathResources;
import static org.mule.connectivity.restconnect.internal.webapi.util.XmlUtils.getXmlName;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorOperation;
import org.mule.connectivity.restconnect.internal.connectormodel.type.schema.CustomTypeSchema;
import org.mule.connectivity.restconnect.internal.connectormodel.type.schema.JsonTypeSchema;
import org.mule.connectivity.restconnect.internal.connectormodel.type.schema.TypeSchema;
import org.mule.connectivity.restconnect.internal.connectormodel.type.schema.XmlTypeSchema;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.json.JSONObject;

public class FileGenerationUtils {

  private static final String JSON_REF_PROPERTY = "$ref";
  private static final Pattern TYPE_DEF_NAME_PATTERN = compile("#/definitions/(.*)");

  public static String writeSchema(TypeSchema source, Path outputDir, ConnectorOperation operation, SchemaNameType nameType,
                                   String partName, Map<TypeSchema, String> typeSchemaPaths) {
    String existentPath = typeSchemaPaths.get(source);

    String schemaName = generateSchemaName(source, operation, nameType, partName, outputDir);

    if (existentPath != null) {
      return existentPath;
    } else {
      try {
        String schemaString = source.getRawSchema();

        if (source instanceof JsonTypeSchema) {
          JSONObject jObject = new JSONObject(schemaString);
          schemaString = jObject.toString(2);
        }
        writeStringToFile(outputDir.resolve(schemaName).toFile(), schemaString, UTF_8);

        typeSchemaPaths.put(source, schemaName);

        return schemaName;

      } catch (IOException e) {
        throw new IllegalArgumentException(format("Path %s is invalid.", outputDir), e);
      }
    }
  }

  public enum SchemaNameType {
    INPUT, OUTPUT, PART
  }

  private static String generateSchemaName(TypeSchema source, ConnectorOperation operation, SchemaNameType nameType,
                                           String partName, Path outputDir) {

    String jsonRef = getJsonRefProperty(source);
    if (jsonRef != null) {
      String refSchemaName = getXmlName(jsonRef) + "." + getExtension(source);
      if (!outputDir.resolve(refSchemaName).toFile().exists()) {
        return refSchemaName;
      }
    }

    String suffix = getSchemaSuffix(source, nameType, partName);

    String resourcesSchemaName = (join("-", getPathResources(operation.getPath())) + suffix).toLowerCase();
    if (!outputDir.resolve(resourcesSchemaName).toFile().exists()) {
      return resourcesSchemaName;
    }

    String operationSchemaName = operation.getInternalName() + suffix;
    if (!outputDir.resolve(operationSchemaName).toFile().exists()) {
      return operationSchemaName;
    }

    throw new IllegalArgumentException("Could not generate name for schema.");
  }

  private static String getJsonRefProperty(TypeSchema source) {
    if (source instanceof JsonTypeSchema || source instanceof CustomTypeSchema) {
      JSONObject jsonSchema = new JSONObject(source.getRawSchema());
      if (jsonSchema.keySet().contains(JSON_REF_PROPERTY)) {
        String ref = jsonSchema.getString(JSON_REF_PROPERTY);
        Matcher refMatcher = TYPE_DEF_NAME_PATTERN.matcher(ref);
        if (refMatcher.find()) {
          String name = refMatcher.group(1);
          if (!name.equalsIgnoreCase("root") && !name.equalsIgnoreCase("schema")) {
            return name;
          }
        }
      }
    }
    return null;
  }

  private static String getSchemaSuffix(TypeSchema source, SchemaNameType nameType, String partName) {
    String schemaSuffix = EMPTY;

    switch (nameType) {
      case INPUT:
        schemaSuffix = "-input-schema.";
        break;
      case OUTPUT:
        schemaSuffix = "-output-schema.";
        break;
      case PART:
        schemaSuffix = "-part-input-schema.";
        break;
    }

    if (isNotBlank(partName)) {
      schemaSuffix = "-" + getXmlName(partName) + schemaSuffix;
    }

    return schemaSuffix + getExtension(source);
  }

  private static String getExtension(TypeSchema source) {
    if (source instanceof XmlTypeSchema) {
      return "xsd";
    }

    if (source instanceof JsonTypeSchema || source instanceof CustomTypeSchema) {
      return "json";
    }

    throw new IllegalArgumentException("Type Schema class doesn't support typeSchema generation.");
  }
}
