/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.connectormodel.builder;

import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.connectivity.restconnect.internal.connectormodel.Protocol.HTTP;
import static org.mule.connectivity.restconnect.internal.connectormodel.Protocol.HTTPS;
import static org.mule.connectivity.restconnect.internal.connectormodel.QueryParamArrayFormat.MULTIMAP;
import static org.mule.connectivity.restconnect.internal.webapi.util.XmlUtils.getXmlName;
import static org.mule.connectivity.restconnect.internal.webapi.util.XmlUtils.removeMavenArtifactUnwantedCharacters;
import static org.mule.connectivity.restconnect.internal.webapi.util.XmlUtils.removeMavenGroupUnwantedCharacters;
import static org.mule.connectivity.restconnect.internal.webapi.util.XmlUtils.removeMavenVersionUnwantedCharacters;

import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.exception.UnsupportedPaginationParameterException;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorCategory;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorModel;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorPackage;
import org.mule.connectivity.restconnect.internal.connectormodel.Protocol;
import org.mule.connectivity.restconnect.internal.connectormodel.QueryParamArrayFormat;
import org.mule.connectivity.restconnect.internal.connectormodel.pagination.InPaginationParameter;
import org.mule.connectivity.restconnect.internal.connectormodel.pagination.OutPaginationParameter;
import org.mule.connectivity.restconnect.internal.connectormodel.pagination.Pagination;
import org.mule.connectivity.restconnect.internal.connectormodel.pagination.PaginationParameter;
import org.mule.connectivity.restconnect.internal.connectormodel.pagination.ResponseValueExtraction;
import org.mule.connectivity.restconnect.internal.connectormodel.uri.BaseUri;
import org.mule.connectivity.restconnect.internal.descriptor.model.ConnectorDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.InPaginationParameterDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.OutPaginationParameterDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.PaginationDeclarationDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.PaginationParameterDescriptor;
import org.mule.connectivity.restconnect.internal.util.AMFWrapper;
import org.mule.connectivity.restconnect.internal.webapi.model.APIModel;
import org.mule.connectivity.restconnect.internal.webapi.parser.TypeSchemaPool;

import java.util.ArrayList;
import java.util.List;

public class ConnectorModelBuilder {

  private final static List<Protocol> DEFAULT_PROTOCOLS;

  static {
    DEFAULT_PROTOCOLS = new ArrayList<>();
    DEFAULT_PROTOCOLS.add(HTTP);
    DEFAULT_PROTOCOLS.add(HTTPS);
  }

  private final ConnectorOperationBuilder connectorOperationBuilder;

  public ConnectorModelBuilder() {
    this.connectorOperationBuilder = new ConnectorOperationBuilder(new TypeSchemaPool());
  }

  public ConnectorModel buildConnectorModel(APIModel apiModel, ConnectorDescriptor connectorDescriptor)
      throws ModelGenerationException {
    AMFWrapper.initialize();

    List<Protocol> protocols = buildProtocols(apiModel);

    return new ConnectorModel(buildConnectorName(connectorDescriptor),
                              connectorDescriptor.getConnectorDescription(),
                              buildMavenGroup(connectorDescriptor),
                              buildMavenArtifactId(connectorDescriptor),
                              buildMavenVersion(connectorDescriptor),
                              buildBasePackage(connectorDescriptor),
                              buildConnectorCategory(connectorDescriptor),
                              buildBaseUri(connectorDescriptor, apiModel, protocols),
                              protocols,
                              connectorOperationBuilder.buildOperations(apiModel, connectorDescriptor),
                              buildPaginations(connectorDescriptor),
                              buildExtensionXml(connectorDescriptor),
                              buildSkipOutputTypeValidation(connectorDescriptor),
                              buildQueryParamArrayFormat(connectorDescriptor));
  }

  protected QueryParamArrayFormat buildQueryParamArrayFormat(ConnectorDescriptor connectorDescriptor) {
    if (isNotBlank(connectorDescriptor.getQueryParamArrayFormat())) {
      return QueryParamArrayFormat.valueOf(connectorDescriptor.getQueryParamArrayFormat().toUpperCase());
    }
    return MULTIMAP;
  }

  private static String buildConnectorName(ConnectorDescriptor connectorDescriptor) {
    String connectorName = connectorDescriptor.getConnectorName().trim();
    if (connectorName.toLowerCase().endsWith("api")) {
      connectorName = connectorName.substring(0, connectorName.lastIndexOf("api"));
    }

    if (connectorName.toLowerCase().endsWith("connector")) {
      return connectorName;
    }

    return connectorName + " Connector";
  }

  private static Boolean buildSkipOutputTypeValidation(ConnectorDescriptor connectorDescriptor) {
    return connectorDescriptor.getSkipOutputTypeValidation();
  }

  private static String buildExtensionXml(ConnectorDescriptor connectorDescriptor) {
    if (isNotBlank(connectorDescriptor.getExtensionXml())) {
      return connectorDescriptor.getExtensionXml();
    }
    return getXmlName(connectorDescriptor.getConnectorName());
  }

  private static String buildBasePackage(ConnectorDescriptor connectorDescriptor) {
    if (isNotBlank(connectorDescriptor.getBaseJavaPackage())) {
      return connectorDescriptor.getBaseJavaPackage();
    }
    return ConnectorPackage.buildBasePackage(connectorDescriptor.getConnectorName());
  }

  private static List<Protocol> buildProtocols(APIModel apiModel) {
    return apiModel.getProtocols().isEmpty() ? DEFAULT_PROTOCOLS : apiModel.getProtocols();
  }

  private static String buildMavenGroup(ConnectorDescriptor connectorDescriptor) {
    return removeMavenGroupUnwantedCharacters(connectorDescriptor.getConnectorGav().getGroupId());
  }

  private static String buildMavenArtifactId(ConnectorDescriptor connectorDescriptor) {
    return removeMavenArtifactUnwantedCharacters(connectorDescriptor.getConnectorGav().getArtifactId());
  }

  private static String buildMavenVersion(ConnectorDescriptor connectorDescriptor) {
    return removeMavenVersionUnwantedCharacters(connectorDescriptor.getConnectorGav().getVersion());
  }

  private static ConnectorCategory buildConnectorCategory(ConnectorDescriptor connectorDescriptor) {
    return ConnectorCategory.fromString(connectorDescriptor.getConnectorCategory());
  }

  private static BaseUri buildBaseUri(ConnectorDescriptor connectorDescriptor, APIModel apiModel,
                                      List<Protocol> supportedProtocols) {
    if (connectorDescriptor.getBaseUri() == null) {
      return apiModel.getBaseUri();
    }

    BaseUri generatedUri = new BaseUri(connectorDescriptor.getBaseUri().getValue(),
                                       BaseUri.Type.valueOfName(connectorDescriptor.getBaseUri().getType()),
                                       apiModel.getBaseUri().getApiVersion(),
                                       supportedProtocols);

    connectorDescriptor.getBaseUri().getMapping()
        .forEach(x -> generatedUri.addMultipleBaseUri(x.getName(), x.getValue(), x.isDefault()));

    return generatedUri;
  }

  private static List<Pagination> buildPaginations(ConnectorDescriptor connectorDescriptor)
      throws UnsupportedPaginationParameterException {
    List<Pagination> paginations = new ArrayList<>(connectorDescriptor.getPaginations().size());
    for (PaginationDeclarationDescriptor paginationDescriptor : connectorDescriptor.getPaginations()) {
      paginations.add(new Pagination(paginationDescriptor.getName(), paginationDescriptor.getType(),
                                     getPagingDescriptorParameters(paginationDescriptor.getParameters())));
    }

    return paginations;
  }

  private static List<PaginationParameter> getPagingDescriptorParameters(List<PaginationParameterDescriptor> pagingParameters)
      throws UnsupportedPaginationParameterException {
    List<PaginationParameter> parameters = new ArrayList<>(pagingParameters.size());
    for (PaginationParameterDescriptor parameterDescriptor : pagingParameters) {
      PaginationParameter paginationParameter;
      if (parameterDescriptor instanceof InPaginationParameterDescriptor) {
        ResponseValueExtraction valueExtraction =
            new ResponseValueExtraction(((InPaginationParameterDescriptor) parameterDescriptor).getValueExtraction()
                .getLanguage(), ((InPaginationParameterDescriptor) parameterDescriptor).getValueExtraction().getContent());
        paginationParameter = new InPaginationParameter(parameterDescriptor.getName(), valueExtraction);
      } else if (parameterDescriptor instanceof OutPaginationParameterDescriptor) {
        paginationParameter = new OutPaginationParameter(parameterDescriptor.getName(),
                                                         ((OutPaginationParameterDescriptor) parameterDescriptor).getValue());
      } else {
        throw new UnsupportedPaginationParameterException("Only In/Out pagination parameters are supported.");
      }
      parameters.add(paginationParameter);
    }

    return parameters;
  }
}
