/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.connectormodel.builder;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isBlank;
import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.connectormodel.security.ConnectorSecurityScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.BasicAuthScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.CustomAuthenticationScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.DigestAuthenticationScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.NotSupportedScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.OAuth2AuthorizationCodeScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.OAuth2ClientCredentialsScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.PassThroughScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.security.TestConnectionConfig;
import org.mule.connectivity.restconnect.internal.connectormodel.security.TestConnectionValidationConfig;
import org.mule.connectivity.restconnect.internal.connectormodel.security.UnsecuredScheme;
import org.mule.connectivity.restconnect.internal.descriptor.model.ConnectorDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.SecuritySchemeDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.TestConnectionDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.TestConnectionResponseValidationDescriptor;
import org.mule.connectivity.restconnect.internal.webapi.model.APISecuritySchemeModel;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.http.api.HttpConstants.Method;

import java.util.LinkedList;
import java.util.List;

public class ConnectorSecuritySchemeBuilder {

  private final ConnectorParameterBuilder parameterBuilder;

  public ConnectorSecuritySchemeBuilder(ConnectorTypeDefinitionBuilder connectorTypeDefinitionBuilder) {
    this.parameterBuilder = new ConnectorParameterBuilder(connectorTypeDefinitionBuilder);
  }

  public List<ConnectorSecurityScheme> buildSecuritySchemes(List<APISecuritySchemeModel> apiSecuritySchemeModels,
                                                            ConnectorDescriptor connectorDescriptor)
      throws ModelGenerationException {
    List<SecuritySchemeDescriptor> securityDescriptors = connectorDescriptor.getSecurity();
    List<ConnectorSecurityScheme> apiSecuritySchemes = new LinkedList<>();

    if (!apiSecuritySchemeModels.isEmpty()) {
      for (APISecuritySchemeModel securitySchemeModel : apiSecuritySchemeModels) {
        ConnectorSecurityScheme apiSecurityScheme =
            buildSecurityScheme(securitySchemeModel, getSecuritySchemeDescriptor(securityDescriptors, securitySchemeModel),
                                connectorDescriptor);
        if (apiSecurityScheme == null) {
          continue;
        }
        if (!(apiSecurityScheme instanceof NotSupportedScheme)) {
          apiSecuritySchemes.add(apiSecurityScheme);
        }
      }
    } else {
      apiSecuritySchemes.add(new UnsecuredScheme(buildTestConnectionConfig(connectorDescriptor.getTestConnection())));
    }

    return apiSecuritySchemes;
  }

  private TestConnectionConfig buildTestConnectionConfig(TestConnectionDescriptor testConnectionDescriptor) {
    if (testConnectionDescriptor == null) {
      return null;
    }

    return new TestConnectionConfig(testConnectionDescriptor.getPath(),
                                    testConnectionDescriptor.getMethod() == null ? null
                                        : Method.valueOf(testConnectionDescriptor.getMethod().toUpperCase()),
                                    testConnectionDescriptor.getValidStatusCodes(),
                                    testConnectionDescriptor.getMediaType() == null ? null
                                        : MediaType.parse(testConnectionDescriptor.getMediaType()),
                                    testConnectionDescriptor.getResponseValidation().stream()
                                        .map(this::buildTestConnectionValidationConfig)
                                        .collect(toList()));
  }

  private TestConnectionValidationConfig buildTestConnectionValidationConfig(TestConnectionResponseValidationDescriptor responseValidation) {
    if (responseValidation.getValidationExpression() == null
        || isBlank(responseValidation.getValidationExpression().getContent())) {
      return null;
    }

    String errorExpression = responseValidation.getErrorTemplateExpression() != null
        ? responseValidation.getErrorTemplateExpression().getContent() : null;

    return new TestConnectionValidationConfig(responseValidation.getValidationExpression().getContent(), errorExpression);
  }

  private SecuritySchemeDescriptor getSecuritySchemeDescriptor(List<SecuritySchemeDescriptor> securities,
                                                               APISecuritySchemeModel securitySchemeModel) {
    return securities.stream()
        .filter(x -> x.getDisplayName().equalsIgnoreCase(securitySchemeModel.getName()))
        .findFirst().orElse(null);
  }

  private ConnectorSecurityScheme buildSecurityScheme(APISecuritySchemeModel securitySchemeModel,
                                                      SecuritySchemeDescriptor securitySchemeDescriptor,
                                                      ConnectorDescriptor connectorDescriptor)
      throws ModelGenerationException {

    TestConnectionDescriptor testConnectionDescriptor =
        (securitySchemeDescriptor != null && securitySchemeDescriptor.getTestConnection() != null)
            ? securitySchemeDescriptor.getTestConnection() : connectorDescriptor.getTestConnection();
    TestConnectionConfig testConnectionConfig = buildTestConnectionConfig(testConnectionDescriptor);

    if (securitySchemeDescriptor != null && securitySchemeDescriptor.isIgnored()) {
      return null;
    }
    Class<?> securitySchemeClass = securitySchemeModel.getSecuritySchemeClass();
    List<String> collisionNames = new LinkedList<>();

    List<Parameter> customQueryParams =
        parameterBuilder.buildParameterList(securitySchemeModel.getCustomQueryParams(), securitySchemeDescriptor, collisionNames);
    collisionNames.addAll(customQueryParams.stream().map(Parameter::getInternalName).collect(toList()));
    List<Parameter> customHeaders =
        parameterBuilder.buildParameterList(securitySchemeModel.getCustomHeaders(), securitySchemeDescriptor, collisionNames);

    ConnectorSecurityScheme.Builder builder =
        ConnectorSecurityScheme.builder(securitySchemeModel.getName(), customQueryParams, customHeaders, testConnectionConfig);

    if (securitySchemeClass.equals(BasicAuthScheme.class)) {
      return builder.buildBasicAuthScheme();
    } else if (securitySchemeClass.equals(CustomAuthenticationScheme.class)) {
      return builder.buildCustomAuthenticationScheme();
    } else if (securitySchemeClass.equals(DigestAuthenticationScheme.class)) {
      return builder.buildDigestAuthenticationSchemeScheme();
    } else if (securitySchemeClass.equals(OAuth2AuthorizationCodeScheme.class)) {
      return builder.buildOAuth2AuthorizationCodeScheme(securitySchemeModel.getAuthorizationUri(),
                                                        securitySchemeModel.getAccessTokenUri(),
                                                        securitySchemeModel.getScopes());
    } else if (securitySchemeClass.equals(OAuth2ClientCredentialsScheme.class)) {
      return builder.buildOAuth2ClientCredentialsScheme(securitySchemeModel.getAuthorizationUri(),
                                                        securitySchemeModel.getAccessTokenUri(),
                                                        securitySchemeModel.getScopes());
    } else if (securitySchemeClass.equals(PassThroughScheme.class)) {
      return builder.buildPassThroughScheme();
    } else {
      return builder.buildUnsecuredScheme();
    }
  }
}

