/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.descriptor.parser;

import static java.util.Collections.emptyList;

import org.mule.connectivity.restconnect.internal.descriptor.model.InPaginationParameterDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.OutPaginationParameterDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.PaginationDeclarationDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.PaginationParameterDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.ExpressionDescriptor;

import java.util.ArrayList;
import java.util.List;

import javax.validation.constraints.NotNull;

import amf.client.model.domain.DialectDomainElement;

public class DescriptorPaginationParser {

  private static final String AML_REST_CONNECT_PAGINATION_DECLARATIONS =
      "http://a.ml/vocabularies/restConnect#paginationDeclarations";
  private static final String AML_REST_CONNECT_PAGINATION_DECLARATION = "http://a.ml/vocabularies.restc/core#name";
  private static final String AML_REST_CONNECT_PAGINATION_TYPE = "http://a.ml/vocabularies/restConnect#paginationType";
  private static final String AML_REST_CONNECT_PAGINATION_PARAMETERS =
      "http://a.ml/vocabularies/restConnect#paginationParameters";
  private static final String AML_REST_CONNECT_PAGINATION_OFFSET_PARAM_NAME =
      "http://a.ml/vocabularies/restConnect#offsetParamName";
  private static final String AML_REST_CONNECT_PAGINATION_INITIAL_OFFSET = "http://a.ml/vocabularies/restConnect#initialOffset";
  private static final String AML_REST_CONNECT_PAGINATION_NEXT_TOKEN_PARAM_NAME =
      "http://a.ml/vocabularies/restConnect#nextTokenParamName";
  private static final String AML_REST_CONNECT_PAGINATION_NEXT_TOKEN_EXPRESSION =
      "http://a.ml/vocabularies/restConnect#nextTokenExpression";
  private static final String AML_REST_CONNECT_PAGINATION_PAGE_NUMBER_PARAM_NAME =
      "http://a.ml/vocabularies/restConnect#pageNumberParamName";
  private static final String AML_REST_CONNECT_PAGINATION_INITIAL_PAGE_NUMBER =
      "http://a.ml/vocabularies/restConnect#initialPageNumber";
  private static final String AML_REST_CONNECT_PAGINATION_PAGING_RESPONSE_EXPRESSION =
      "http://a.ml/vocabularies/restConnect#pagingResponseExpression";

  private static DescriptorExpressionParser expressionParser = new DescriptorExpressionParser();

  private DescriptorPaginationParser() {}

  public static List<PaginationDeclarationDescriptor> parsePaginationsDescription(DialectDomainElement encodes) {
    List<DialectDomainElement> paginationsDescriptor = encodes.getObjectPropertyUri(AML_REST_CONNECT_PAGINATION_DECLARATIONS);

    return paginationsDescriptor.isEmpty() ? emptyList() : getPagingMethods(paginationsDescriptor);
  }

  private static List<PaginationDeclarationDescriptor> getPagingMethods(List<DialectDomainElement> paginationsDescriptor) {
    List<PaginationDeclarationDescriptor> paginationDeclarationDescriptors = new ArrayList<>(paginationsDescriptor.size());
    for (DialectDomainElement endpointsElement : paginationsDescriptor) {
      String pagingStrategyName = parsePagingStrategyName(endpointsElement);
      String pagingType = parsePagingType(endpointsElement);

      List<DialectDomainElement> parameters = endpointsElement.getObjectPropertyUri(AML_REST_CONNECT_PAGINATION_PARAMETERS);
      List<PaginationParameterDescriptor> paginationParameterDescriptors = new ArrayList<>(parameters.size());
      if (!parameters.isEmpty()) {
        DialectDomainElement pagingParameterItem = parameters.get(0);
        OutPaginationParameterDescriptor offsetParamNameDescriptor = parseOffsetParamName(pagingParameterItem);
        if (offsetParamNameDescriptor != null) {
          paginationParameterDescriptors.add(offsetParamNameDescriptor);
        }

        OutPaginationParameterDescriptor initialOffsetDescriptor = parseInitialOffset(pagingParameterItem);
        if (initialOffsetDescriptor != null) {
          paginationParameterDescriptors.add(initialOffsetDescriptor);
        }

        OutPaginationParameterDescriptor nextTokenParamName = parseNextTokenParamName(pagingParameterItem);
        if (nextTokenParamName != null) {
          paginationParameterDescriptors.add(nextTokenParamName);
        }

        InPaginationParameterDescriptor nextTokenExpression = parseNextTokenExpression(pagingParameterItem);
        if (nextTokenExpression != null) {
          paginationParameterDescriptors.add(nextTokenExpression);
        }

        OutPaginationParameterDescriptor pageNumberParamName =
            parsePageNumberParamName(pagingParameterItem);
        if (pageNumberParamName != null) {
          paginationParameterDescriptors.add(pageNumberParamName);
        }

        OutPaginationParameterDescriptor initialPageNumber =
            parseInitialPageNumber(pagingParameterItem);
        if (initialPageNumber != null) {
          paginationParameterDescriptors.add(initialPageNumber);
        }

        InPaginationParameterDescriptor pagingResponseExpression =
            parsePagingResponseExpression(pagingParameterItem);
        if (pagingResponseExpression != null) {
          paginationParameterDescriptors.add(pagingResponseExpression);
        }
      }

      paginationDeclarationDescriptors
          .add(new PaginationDeclarationDescriptor(pagingStrategyName, pagingType, paginationParameterDescriptors));
    }

    return paginationDeclarationDescriptors;
  }

  private static OutPaginationParameterDescriptor parseOffsetParamName(@NotNull DialectDomainElement pagingParameterItem) {
    List<Object> offsetParamName = pagingParameterItem.getScalarByPropertyUri(AML_REST_CONNECT_PAGINATION_OFFSET_PARAM_NAME);

    return offsetParamName.isEmpty() ? null
        : new OutPaginationParameterDescriptor("offsetParamName", offsetParamName.get(0).toString());
  }

  private static OutPaginationParameterDescriptor parseInitialOffset(@NotNull DialectDomainElement pagingParameterItem) {
    List<Object> initialOffset = pagingParameterItem.getScalarByPropertyUri(AML_REST_CONNECT_PAGINATION_INITIAL_OFFSET);

    return initialOffset.isEmpty() ? null
        : new OutPaginationParameterDescriptor("initialOffset", initialOffset.get(0).toString());
  }

  private static OutPaginationParameterDescriptor parseNextTokenParamName(@NotNull DialectDomainElement pagingParameterItem) {
    List<Object> nextTokenParamName =
        pagingParameterItem.getScalarByPropertyUri(AML_REST_CONNECT_PAGINATION_NEXT_TOKEN_PARAM_NAME);

    return nextTokenParamName.isEmpty() ? null
        : new OutPaginationParameterDescriptor("nextTokenParamName", nextTokenParamName.get(0).toString());
  }

  private static OutPaginationParameterDescriptor parsePageNumberParamName(@NotNull DialectDomainElement pagingParameterItem) {
    List<Object> pageNumberParamName =
        pagingParameterItem.getScalarByPropertyUri(AML_REST_CONNECT_PAGINATION_PAGE_NUMBER_PARAM_NAME);

    return pageNumberParamName.isEmpty() ? null
        : new OutPaginationParameterDescriptor("pageNumberParamName", pageNumberParamName.get(0).toString());
  }

  private static OutPaginationParameterDescriptor parseInitialPageNumber(@NotNull DialectDomainElement pagingParameterItem) {
    List<Object> initialPageNumber = pagingParameterItem.getScalarByPropertyUri(AML_REST_CONNECT_PAGINATION_INITIAL_PAGE_NUMBER);

    return initialPageNumber.isEmpty() ? null
        : new OutPaginationParameterDescriptor("initialPageNumber", initialPageNumber.get(0).toString());
  }

  private static InPaginationParameterDescriptor parseNextTokenExpression(@NotNull DialectDomainElement pagingParameterItem) {
    List<DialectDomainElement> nextTokenExpression =
        pagingParameterItem.getObjectPropertyUri(AML_REST_CONNECT_PAGINATION_NEXT_TOKEN_EXPRESSION);
    ExpressionDescriptor expressionDescriptor =
        parseResponseValueExtractionDescriptor(nextTokenExpression);

    return expressionDescriptor == null ? null
        : new InPaginationParameterDescriptor("nextTokenExpression", expressionDescriptor);
  }

  private static InPaginationParameterDescriptor parsePagingResponseExpression(@NotNull DialectDomainElement pagingParameterItem) {
    List<DialectDomainElement> pagingResponseExpression =
        pagingParameterItem.getObjectPropertyUri(AML_REST_CONNECT_PAGINATION_PAGING_RESPONSE_EXPRESSION);
    ExpressionDescriptor expressionDescriptor =
        parseResponseValueExtractionDescriptor(pagingResponseExpression);

    return expressionDescriptor == null ? null
        : new InPaginationParameterDescriptor("pagingResponseExpression", expressionDescriptor);
  }

  private static String parsePagingStrategyName(@NotNull DialectDomainElement endpointsElement) {
    List<Object> pagingStrategyNames = endpointsElement.getScalarByPropertyUri(AML_REST_CONNECT_PAGINATION_DECLARATION);

    return pagingStrategyNames.isEmpty() ? null : pagingStrategyNames.get(0).toString();
  }

  private static String parsePagingType(@NotNull DialectDomainElement endpointsElement) {
    List<Object> pagingTypes = endpointsElement.getScalarByPropertyUri(AML_REST_CONNECT_PAGINATION_TYPE);

    return pagingTypes.isEmpty() ? null : pagingTypes.get(0).toString();
  }

  private static ExpressionDescriptor parseResponseValueExtractionDescriptor(List<DialectDomainElement> valueExtraction) {
    if (!valueExtraction.isEmpty()) {
      return expressionParser.parseExpression(valueExtraction.get(0));
    }

    return null;
  }
}
