/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.descriptor.parser;

import static java.lang.String.format;
import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.internal.descriptor.model.RequestDescriptor;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.ParameterType;

import com.google.common.collect.ImmutableMap;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import amf.client.model.domain.DialectDomainElement;

public class DescriptorRequestParser {

  private static final Map<String, String> AMF_PARAMETER_TYPE = ImmutableMap.<String, String>builder()
      .put(ParameterType.URI.getName(), "uriParameter")
      .put(ParameterType.QUERY.getName(), "queryString")
      .put(ParameterType.HEADER.getName(), "header")
      .build();

  private static final String API_CONTRACT_EXPECTS = "http://a.ml/vocabularies.restc/apiContract#expects";
  private static final String API_CONTRACT_PARAMETER_NAMES = "http://a.ml/vocabularies.restc/apiContract#%s";
  private static final String REST_CONNECT_PART = "http://a.ml/vocabularies/restConnect#part";
  private DescriptorParameterParser descriptorParameterParser = new DescriptorParameterParser();
  private DescriptorPartParser descriptorPartParser = new DescriptorPartParser();

  public RequestDescriptor parseRequest(DialectDomainElement operationElement, Path descriptorFilePath)
      throws ModelGenerationException {
    DialectDomainElement expectsElement = parseExpectsElement(operationElement);

    if (expectsElement == null) {
      return null;
    }

    return new RequestDescriptor(
                                 descriptorParameterParser
                                     .parseParameters(getParametersByType(expectsElement, ParameterType.HEADER)),
                                 descriptorParameterParser
                                     .parseParameters(getParametersByType(expectsElement, ParameterType.QUERY)),
                                 descriptorParameterParser
                                     .parseParameters(getParametersByType(expectsElement, ParameterType.URI)),
                                 descriptorPartParser.parseParts(getParts(expectsElement), descriptorFilePath));
  }


  private List<DialectDomainElement> getParametersByType(DialectDomainElement expects, ParameterType parameterType) {
    String amfParameterType = format(API_CONTRACT_PARAMETER_NAMES, AMF_PARAMETER_TYPE.get(parameterType.getName()));
    return new ArrayList<>(expects.getObjectPropertyUri(amfParameterType));
  }

  private List<DialectDomainElement> getParts(DialectDomainElement expects) {
    return new ArrayList<>(expects.getObjectPropertyUri(REST_CONNECT_PART));
  }

  private DialectDomainElement parseExpectsElement(DialectDomainElement operationElement) {
    List<DialectDomainElement> expectsElements = operationElement.getObjectPropertyUri(API_CONTRACT_EXPECTS);
    return expectsElements.isEmpty() ? null : expectsElements.get(0);
  }
}
