/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.descriptor.writer;

import static org.apache.commons.lang3.StringUtils.EMPTY;
import org.mule.connectivity.restconnect.internal.descriptor.model.APIUrlDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.BaseUriDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.DescriptorElement;
import org.mule.connectivity.restconnect.internal.descriptor.model.EndPointDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.MavenGavDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.MultipleBaseUriDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.OperationDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.ParameterDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.RequestDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.ConnectorDescriptor;

import java.util.List;

public class DescriptorWriter {

  private final static String DESCRIPTOR_HEADER = "#% Rest Connect Connector Descriptor 1.0";
  private final static String EMPTY_LINE = EMPTY;

  private final static String API_SPEC_KEY = "apiSpec";
  private final static String CONNECTOR_NAME_KEY = "connectorName";
  private final static String CONNECTOR_DESCRIPTION_KEY = "connectorDescription";
  private final static String CONNECTOR_GAV_KEY = "connectorGav";
  private final static String SKIP_OUTPUT_TYPE_VALIDATION_KEY = "skipOutputTypeValidation";
  private final static String VOID_OPERATION_KEY = "voidOperation";
  private final static String BASE_URI_KEY = "baseUri";
  private final static String PAGINATIONS_KEY = "paginations";
  private final static String ENDPOINTS_KEY = "endpoints";
  private final static String SECURITY_KEY = "security";

  private final static String GROUP_KEY = "groupId";
  private final static String ARTIFACT_KEY = "artifactId";
  private final static String VERSION_KEY = "version";

  private final static String URL_KEY = "url";

  private final static String BASE_URI_VALUE_KEY = "value";
  private final static String BASE_URI_TYPE_KEY = "type";
  private final static String BASE_URI_MAPPING_KEY = "mapping";

  private final static String MULTIPLE_BASE_URI_VALUE_KEY = "value";
  private final static String MULTIPLE_BASE_URI_DEFAULT_KEY = "default";

  private final static String ENDPOINT_OPERATIONS_KEY = "operations";
  private final static String ENDPOINT_IGNORED_KEY = "ignored";

  private final static String OPERATION_NAME_KEY = "name";
  private final static String OPERATION_DESCRIPTION_KEY = "description";
  private final static String OPERATION_DEFAULT_INPUT_MEDIA_TYPE_KEY = "inputMediaType";
  private final static String OPERATION_DEFAULT_OUTPUT_MEDIA_TYPE_KEY = "outputMediaType";
  private final static String OPERATION_EXPECTS_KEY = "expects";
  private final static String OPERATION_IGNORED_KEY = "ignored";

  private final static String REQUEST_HEADER_KEY = "header";
  private final static String REQUEST_QUERY_PARAM_KEY = "queryParameter";
  private final static String REQUEST_URI_PARAM_KEY = "uriParameter";

  private final static String PARAMETER_FRIENDLY_NAME_KEY = "friendlyName";
  private final static String PARAMETER_DESCRIPTION_KEY = "description";

  private YamlWriter yamlWriter = new YamlWriter();

  private String getBoolean(boolean bool) {
    return bool ? "true" : "false";
  }

  private void write(DescriptorElement descriptorElement) {
    if (descriptorElement instanceof ParameterDescriptor) {
      write((ParameterDescriptor) descriptorElement);
    } else if (descriptorElement instanceof OperationDescriptor) {
      write((OperationDescriptor) descriptorElement);
    } else if (descriptorElement instanceof EndPointDescriptor) {
      write((EndPointDescriptor) descriptorElement);
    } else if (descriptorElement instanceof MultipleBaseUriDescriptor) {
      write((MultipleBaseUriDescriptor) descriptorElement);
    } else {
      throw new IllegalArgumentException("Can not write generic element");
    }
  }

  public String write(ConnectorDescriptor connectorDescriptor) {
    yamlWriter.addLine(DESCRIPTOR_HEADER).addLine(EMPTY_LINE);

    if (connectorDescriptor.getApiSpec() instanceof MavenGavDescriptor) {
      write((MavenGavDescriptor) connectorDescriptor.getApiSpec(), API_SPEC_KEY);
    } else {
      write((APIUrlDescriptor) connectorDescriptor.getApiSpec(), API_SPEC_KEY);
    }

    yamlWriter
        .addLine(EMPTY_LINE)
        .addKeyValueLine(CONNECTOR_NAME_KEY, connectorDescriptor.getConnectorName())
        .addKeyValueLineIfNotEmpty(CONNECTOR_DESCRIPTION_KEY, connectorDescriptor.getConnectorDescription())
        .addLine(EMPTY_LINE);

    if (connectorDescriptor.getSkipOutputTypeValidation() != null) {
      yamlWriter.addKeyValueLine(SKIP_OUTPUT_TYPE_VALIDATION_KEY,
                                 getBoolean(connectorDescriptor.getSkipOutputTypeValidation()),
                                 "Set this to true if you want to add the type schema manually for all of the operations")
          .addLine(EMPTY_LINE);
    }

    write(connectorDescriptor.getConnectorGav(), CONNECTOR_GAV_KEY);

    yamlWriter.addLine(EMPTY_LINE);
    write(connectorDescriptor.getBaseUri());

    if (!connectorDescriptor.getEndpoints().isEmpty()) {
      yamlWriter.addLine(EMPTY_LINE);
      writeList(connectorDescriptor.getEndpoints(), ENDPOINTS_KEY);
    }

    return yamlWriter.toString();
  }

  public void write(MavenGavDescriptor mavenGavDescriptor, String key) {
    yamlWriter
        .addKeyLine(key)
        .addIndentation()
        .addKeyValueLine(GROUP_KEY, mavenGavDescriptor.getGroupId())
        .addKeyValueLine(ARTIFACT_KEY, mavenGavDescriptor.getArtifactId())
        .addKeyValueLine(VERSION_KEY, mavenGavDescriptor.getVersion())
        .removeIndentation();
  }

  public void write(APIUrlDescriptor apiUrl, String key) {
    yamlWriter
        .addKeyLine(key)
        .addIndentation()
        .addKeyValueLine(URL_KEY, apiUrl.getUrl())
        .removeIndentation();
  }

  private void write(BaseUriDescriptor baseUri) {
    yamlWriter
        .addKeyLine(BASE_URI_KEY)
        .addIndentation();

    if (baseUri.getValue() != null && !baseUri.getValue().isEmpty()) {
      yamlWriter.addKeyValueLine(BASE_URI_VALUE_KEY, baseUri.getValue());
    }

    yamlWriter.addKeyValueLine(BASE_URI_TYPE_KEY, baseUri.getType())
        .removeIndentation();

    writeList(baseUri.getMapping(), BASE_URI_MAPPING_KEY);
  }

  private <T extends DescriptorElement> void writeList(List<T> elements, String key) {
    if (elements != null && !elements.isEmpty()) {
      yamlWriter
          .addKeyLine(key)
          .addIndentation();

      for (T element : elements) {
        write(element);
      }

      yamlWriter.removeIndentation();
    }
  }

  private void write(MultipleBaseUriDescriptor multipleBaseUriDescriptor) {
    yamlWriter
        .addKeyLine(multipleBaseUriDescriptor.getName())
        .addIndentation()
        .addKeyValueLine(MULTIPLE_BASE_URI_VALUE_KEY, multipleBaseUriDescriptor.getValue())
        .addKeyValueLine(MULTIPLE_BASE_URI_DEFAULT_KEY, getBoolean(multipleBaseUriDescriptor.isDefault()))
        .removeIndentation();
  }

  private void write(EndPointDescriptor endPointDescriptor) {
    yamlWriter
        .addKeyLine(endPointDescriptor.getPath())
        .addIndentation()
        .addKeyValueLine(ENDPOINT_IGNORED_KEY, getBoolean(endPointDescriptor.isIgnored()));

    writeList(endPointDescriptor.getOperations(), ENDPOINT_OPERATIONS_KEY);

    yamlWriter.removeIndentation();
  }

  private void write(OperationDescriptor operation) {
    yamlWriter
        .addKeyLine(operation.getMethod())
        .addIndentation()
        .addKeyValueLineIfNotEmpty(OPERATION_NAME_KEY, operation.getName())
        .addKeyValueLineIfNotEmpty(OPERATION_DESCRIPTION_KEY, operation.getDescription())
        .addKeyValueLineIfNotEmpty(OPERATION_DEFAULT_INPUT_MEDIA_TYPE_KEY, operation.getInputMediaType())
        .addKeyValueLineIfNotEmpty(OPERATION_DEFAULT_OUTPUT_MEDIA_TYPE_KEY, operation.getOutputMediaType());

    if (operation.getSkipOutputTypeValidation() != null) {
      yamlWriter.addKeyValueLine(SKIP_OUTPUT_TYPE_VALIDATION_KEY,
                                 getBoolean(operation.getSkipOutputTypeValidation()),
                                 "Set this to true if you want to add the type schema manually");
    }

    if (operation.getVoidOperation() != null) {
      yamlWriter.addKeyValueLine(VOID_OPERATION_KEY,
                                 getBoolean(operation.getVoidOperation()),
                                 "Set this to true if you want this operation response to be empty");
    }

    if (operation.getExpects() != null) {
      write(operation.getExpects());
    }

    yamlWriter
        .addKeyValueLine(OPERATION_IGNORED_KEY, getBoolean(operation.isIgnored()))
        .removeIndentation();
  }

  private void write(RequestDescriptor requestDescriptor) {
    if (!(requestDescriptor.getHeader() == null || requestDescriptor.getHeader().isEmpty()) ||
        !(requestDescriptor.getQueryParameter() == null || requestDescriptor.getQueryParameter().isEmpty()) ||
        !(requestDescriptor.getUriParameter() == null || requestDescriptor.getUriParameter().isEmpty())) {

      yamlWriter
          .addKeyLine(OPERATION_EXPECTS_KEY)
          .addIndentation();

      writeList(requestDescriptor.getHeader(), REQUEST_HEADER_KEY);
      writeList(requestDescriptor.getQueryParameter(), REQUEST_QUERY_PARAM_KEY);
      writeList(requestDescriptor.getUriParameter(), REQUEST_URI_PARAM_KEY);

      yamlWriter
          .removeIndentation();
    }
  }

  private void write(ParameterDescriptor parameterDescriptor) {
    yamlWriter
        .addKeyLine(parameterDescriptor.getParamName())
        .addIndentation()
        .addKeyValueLine(PARAMETER_FRIENDLY_NAME_KEY, parameterDescriptor.getFriendlyName())
        .addKeyValueLineIfNotEmpty(PARAMETER_DESCRIPTION_KEY, parameterDescriptor.getDescription())
        .removeIndentation();
  }

}
