/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.templating.sdk;

import static java.util.stream.Collectors.toList;
import org.mule.connectivity.restconnect.exception.TemplatingException;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorOperation;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorModel;
import org.mule.connectivity.restconnect.internal.templating.JavaTemplateEntity;
import org.mule.connectors.restconnect.commons.api.configuration.RestConnectConfiguration;
import org.mule.runtime.extension.api.annotation.Configuration;
import org.mule.runtime.extension.api.annotation.Operations;
import org.mule.runtime.extension.api.annotation.connectivity.ConnectionProviders;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.TypeSpec;

import java.nio.file.Path;
import java.util.LinkedList;
import java.util.List;

import javax.lang.model.element.Modifier;

public class SdkConfig extends JavaTemplateEntity {

  private final SdkConnector connector;
  private final List<SdkConnectionProvider> connectionProviders;
  private final List<AbstractSdkOperation> operations;

  public SdkConfig(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector) throws TemplatingException {
    super(outputDir, connectorModel);
    this.connector = sdkConnector;

    this.connectionProviders = connectorModel.getOperations().stream()
        .map(ConnectorOperation::getSecuritySchemes)
        .flatMap(List::stream)
        .distinct()
        .map(scheme -> new SdkConnectionProvider(outputDir, connectorModel, sdkConnector, scheme))
        .collect(toList());

    this.operations = new LinkedList<>();
    for (ConnectorOperation operation : connectorModel.getOperations()) {
      this.operations.add(SdkOperationFactory.create(outputDir, connectorModel, sdkConnector, operation));
    }
  }

  public String getJavaClassName() {
    String baseName =
        connector.getJavaName().endsWith("Connector") ? connector.getJavaName().substring(0, connector.getJavaName().length() - 9)
            : connector.getJavaName();

    return baseName + "Configuration";
  }

  public String getPackage() {
    return connectorModel.getBasePackage() + ".internal.config";
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    generateConfigClass();

    for (SdkConnectionProvider connectionProvider : connectionProviders) {
      connectionProvider.applyTemplates();
    }

    for (AbstractSdkOperation operation : operations) {
      operation.applyTemplates();
    }
  }

  private void generateConfigClass() throws TemplatingException {
    TypeSpec.Builder configClassBuilder =
        TypeSpec
            .classBuilder(getJavaClassName())
            .addModifiers(Modifier.PUBLIC)
            .superclass(RestConnectConfiguration.class)
            .addAnnotation(getConfigurationAnnotation())
            .addAnnotation(getOperationsAnnotation())
            .addAnnotation(getConnectionProvidersAnnotation());

    writeClassToFile(configClassBuilder.build(), getPackage());
  }

  private AnnotationSpec getOperationsAnnotation() {
    CodeBlock.Builder codeBlock = CodeBlock.builder();

    codeBlock.add("{");

    for (int i = 0; i < this.operations.size(); i++) {
      AbstractSdkOperation operation = this.operations.get(i);
      codeBlock.add("$T.class", ClassName.get(operation.getPackage(), operation.getJavaClassName()));
      if (i < this.operations.size() - 1) {
        codeBlock.add(",");
      }
    }

    codeBlock.add("}");

    AnnotationSpec.Builder operationsAnnotationBuilder =
        AnnotationSpec
            .builder(Operations.class)
            .addMember(VALUE_MEMBER, codeBlock.build());

    return operationsAnnotationBuilder.build();
  }

  private AnnotationSpec getConnectionProvidersAnnotation() {
    CodeBlock.Builder codeBlock = CodeBlock.builder();

    codeBlock.add("{");

    for (int i = 0; i < this.connectionProviders.size(); i++) {
      SdkConnectionProvider connectionProvider = this.connectionProviders.get(i);
      codeBlock.add("$T.class", ClassName.get(connectionProvider.getPackage(), connectionProvider.getJavaClassName()));
      if (i < this.connectionProviders.size() - 1) {
        codeBlock.add(",");
      }
    }

    codeBlock.add("}");

    AnnotationSpec.Builder connectionProvidersAnnotationBuilder =
        AnnotationSpec
            .builder(ConnectionProviders.class)
            .addMember(VALUE_MEMBER, codeBlock.build());

    return connectionProvidersAnnotationBuilder.build();
  }

  private AnnotationSpec getConfigurationAnnotation() {
    AnnotationSpec.Builder extensionAnnotationBuilder =
        AnnotationSpec
            .builder(Configuration.class);

    return extensionAnnotationBuilder.build();
  }
}
