/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.templating.sdk;

import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.mule.connectivity.restconnect.internal.templating.sdk.SdkContent.SdkContentKind.INPUT_METADATA;
import static org.mule.connectivity.restconnect.internal.util.FileGenerationUtils.SchemaNameType.INPUT;
import static org.mule.connectivity.restconnect.internal.util.FileGenerationUtils.SchemaNameType.PART;
import static org.mule.connectivity.restconnect.internal.util.JavaUtils.getJavaLowerCamelNameFromXml;
import static org.mule.connectivity.restconnect.internal.webapi.util.XmlUtils.getXmlName;
import org.mule.connectivity.restconnect.exception.TemplatingException;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorModel;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorOperation;
import org.mule.connectivity.restconnect.internal.connectormodel.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.templating.JavaTemplateEntity;

import java.io.InputStream;
import java.nio.file.Path;

import org.mule.connectivity.restconnect.internal.util.FileGenerationUtils.SchemaNameType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.annotation.metadata.TypeResolver;
import org.mule.runtime.extension.api.annotation.param.Content;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;

public class SdkContent extends JavaTemplateEntity {

  private final SdkInputMetadataResolver inputMetadataResolver;
  private final String contentParameterJavaName;
  private final boolean primary;

  public enum SdkContentKind {
    INPUT_METADATA, PART
  }

  public SdkContent(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector, ConnectorOperation operation)
      throws TemplatingException {
    this(outputDir, connectorModel, sdkConnector, operation, operation.getInputMetadata(), INPUT_METADATA, EMPTY, true);
  }

  public SdkContent(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector, ConnectorOperation operation,
                    TypeDefinition typeDefinition, SdkContentKind contentType, String partName, boolean primary)
      throws TemplatingException {
    super(outputDir, connectorModel);

    SchemaNameType schemaNameType = contentType.equals(INPUT_METADATA) ? INPUT : PART;

    this.contentParameterJavaName =
        contentType.equals(INPUT_METADATA) ? "content" : getJavaLowerCamelNameFromXml(getXmlName(partName)) + "Content";
    this.primary = primary;

    this.inputMetadataResolver =
        new SdkInputMetadataResolver(outputDir, connectorModel, sdkConnector, operation, typeDefinition, schemaNameType,
                                     partName);
  }

  private TypeName getTypeName() {
    return ParameterizedTypeName.get(TypedValue.class, InputStream.class);
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    inputMetadataResolver.applyTemplates();
  }

  public ParameterSpec generateContentParameter() {
    ParameterSpec.Builder paramSpecBuilder = ParameterSpec
        .builder(getTypeName(), contentParameterJavaName)
        .addAnnotation(generateContentAnnotation());

    if (inputMetadataResolver != null && inputMetadataResolver.getRequiresMetadataResolver()) {
      paramSpecBuilder.addAnnotation(generateMetadataResolverAnnotation());
    }

    return paramSpecBuilder.build();
  }

  private AnnotationSpec generateMetadataResolverAnnotation() {
    return AnnotationSpec
        .builder(TypeResolver.class)
        .addMember("value", "$T.class", ClassName.get(inputMetadataResolver.getPackage(), inputMetadataResolver.getClassName()))
        .build();
  }

  private AnnotationSpec generateContentAnnotation() {
    return AnnotationSpec
        .builder(Content.class)
        .addMember("primary", "$L", primary)
        .build();
  }

  public String getContentParameterJavaName() {
    return contentParameterJavaName;
  }
}
