/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.validation;

import static java.lang.System.lineSeparator;
import static java.util.stream.Collectors.toList;
import static org.mule.connectivity.restconnect.internal.validation.ValidationRule.Level.ERROR;
import static org.mule.connectivity.restconnect.internal.validation.ValidationRule.Level.INFO;
import static org.mule.connectivity.restconnect.internal.validation.ValidationRule.Level.WARN;
import static org.mule.connectivity.restconnect.internal.validation.ModelValidator.defaultModelValidator;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorModel;
import org.mule.connectivity.restconnect.internal.descriptor.model.ConnectorDescriptor;
import org.mule.connectivity.restconnect.internal.webapi.model.APIModel;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ValidationEngine {

  private static Logger logger = LoggerFactory.getLogger(ValidationEngine.class);
  private static final ModelValidator modelValidator = defaultModelValidator();

  private ConnectorDescriptor connectorDescriptor;
  private APIModel apiModel;

  public boolean preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {
    this.connectorDescriptor = connectorDescriptor;
    this.apiModel = apiModel;

    List<ValidationResult> validationResults = modelValidator.preValidate(apiModel, connectorDescriptor);

    reportValidationResults(validationResults);

    return validationResults.stream().noneMatch(x -> x.getLevel().equals(ERROR));
  }

  public boolean postValidate(ConnectorModel connectorModel) {

    List<ValidationResult> validationResults = modelValidator.postValidate(apiModel, connectorDescriptor, connectorModel);

    reportValidationResults(validationResults);

    return validationResults.stream().noneMatch(x -> x.getLevel().equals(ERROR));
  }

  private void reportValidationResults(List<ValidationResult> validationResults) {
    List<ValidationResult> errorList = validationResults.stream()
        .filter(x -> x.getLevel().equals(ERROR))
        .collect(toList());

    List<ValidationResult> warnList = validationResults.stream()
        .filter(x -> x.getLevel().equals(WARN))
        .collect(toList());

    List<ValidationResult> infoList = validationResults.stream()
        .filter(x -> x.getLevel().equals(INFO))
        .collect(toList());

    errorList.forEach(x -> logger.error(x.toString() + lineSeparator()));
    warnList.forEach(x -> logger.warn(x.toString() + lineSeparator()));
    infoList.forEach(x -> logger.warn(x.toString() + lineSeparator()));
  }
}
