/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.validation.rules;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.connectivity.restconnect.internal.validation.ValidationRule.Level.ERROR;
import org.mule.connectivity.restconnect.internal.descriptor.model.ConnectorDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.EndPointDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.OperationDescriptor;
import org.mule.connectivity.restconnect.internal.validation.PreValidationRule;
import org.mule.connectivity.restconnect.internal.validation.ValidationResult;
import org.mule.connectivity.restconnect.internal.webapi.model.APIModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APIOperationModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APITypeModel;

import java.util.LinkedList;
import java.util.List;
import java.util.Optional;

public class DescriptorInputMediaTypeMustBePresentInApiRule extends PreValidationRule {

  //R018
  public DescriptorInputMediaTypeMustBePresentInApiRule() {
    super("The input media type defined in the descriptor is not present in the API spec",
          "If the API spec defines a set of input media types for an operation the Descriptor must use one of those to define the operation's default input media type.",
          ERROR);
  }

  @Override
  public List<ValidationResult> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {

    List<ValidationResult> allErrors = new LinkedList<>();

    for (EndPointDescriptor endpointDesc : connectorDescriptor.getEndpoints()) {
      List<ValidationResult> errors = endpointDesc.getOperations().stream()
          .filter(x -> isNotBlank(x.getInputMediaType()))
          .filter(x -> apiDefinesMediaTypes(apiModel, endpointDesc.getPath(), x))
          .filter(x -> !mediaTypePresentInApi(apiModel, endpointDesc.getPath(), x))
          .map(x -> getValidationErrorForDescriptor(endpointDesc.getPath(), x.getMethod(), x.getInputMediaType()))
          .collect(toList());

      allErrors.addAll(errors);
    }

    String globalMediaType = connectorDescriptor.getDefaultInputMediaType();
    if (isNotBlank(globalMediaType)) {
      List<ValidationResult> globalErrors = apiModel.getOperationsModel().stream()
          .filter(x -> !x.getInputMetadataModel().isEmpty())
          .filter(x -> x.getInputMetadataModel().stream().anyMatch(y -> y.getMediaType() != null))
          .filter(x -> !descriptorDefinesMediaType(connectorDescriptor, x))
          .filter(x -> !descriptorDefinesOperationMediaType(globalMediaType, x))
          .map(x -> getValidationErrorForDescriptor(x.getPath(), x.getHttpMethod().name(), globalMediaType))
          .collect(toList());

      allErrors.addAll(globalErrors);
    }

    return allErrors;
  }

  protected boolean descriptorDefinesOperationMediaType(String globalMediaType, APIOperationModel apiOperationModel) {
    return apiOperationModel.getInputMetadataModel().stream()
        .anyMatch(y -> y.getMediaType().toString().equalsIgnoreCase(globalMediaType));
  }

  private boolean descriptorDefinesMediaType(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperation) {
    return connectorDescriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(apiOperation.getPath()))
        .flatMap(x -> x.getOperations().stream())
        .filter(x -> x.getMethod().equalsIgnoreCase(apiOperation.getHttpMethod().name()))
        .anyMatch(x -> isNotBlank(x.getInputMediaType()));
  }

  private boolean mediaTypePresentInApi(APIModel apiModel, String path, OperationDescriptor operationDesc) {
    String inputMediaType = operationDesc.getInputMediaType();
    String method = operationDesc.getMethod();

    return apiModel.getOperationsModel().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(path) && x.getHttpMethod().name().equalsIgnoreCase(method))
        .flatMap(x -> x.getInputMetadataModel().stream())
        .map(APITypeModel::getMediaType)
        .anyMatch(x -> x.toString().equalsIgnoreCase(inputMediaType));
  }

  private boolean apiDefinesMediaTypes(APIModel apiModel, String path, OperationDescriptor operationDesc) {
    String method = operationDesc.getMethod();

    Optional<APIOperationModel> any = apiModel.getOperationsModel().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(path) && x.getHttpMethod().name().equalsIgnoreCase(method))
        .filter(x -> !x.getInputMetadataModel().isEmpty())
        .findAny();

    return any.isPresent();
  }

  private ValidationResult getValidationErrorForDescriptor(String path, String method, String mediaType) {
    String location =
        "Operation with PATH: " + path + " and METHOD: " + method + " does not declare '" + mediaType + "' media type";
    return new ValidationResult(this, location);
  }
}
