/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.validation.rules;

import static java.util.stream.Collectors.toList;
import static org.mule.connectivity.restconnect.internal.validation.ValidationRule.Level.ERROR;
import org.mule.connectivity.restconnect.internal.descriptor.model.ConnectorDescriptor;
import org.mule.connectivity.restconnect.internal.validation.PreValidationRule;
import org.mule.connectivity.restconnect.internal.validation.ValidationResult;
import org.mule.connectivity.restconnect.internal.webapi.model.APIModel;

import java.util.LinkedList;
import java.util.List;

import org.apache.commons.lang3.tuple.Pair;

public class DescriptorOperationMustExistInApiRule extends PreValidationRule {

  //R002
  public DescriptorOperationMustExistInApiRule() {
    super("Descriptor operations must be present in the API spec",
          "Any operation (PATH + METHOD) declared in the connector descriptor must be present in the API spec.",
          ERROR);
  }

  @Override
  public List<ValidationResult> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {

    List<Pair<String, String>> descriptorOperations =
        connectorDescriptor.getEndpoints().stream()
            .flatMap(x -> x.getOperations().stream().map(y -> Pair.of(x.getPath(), y.getMethod())))
            .collect(toList());

    List<Pair<String, String>> apiOperations = apiModel.getOperationsModel().stream()
        .map(x -> Pair.of(x.getPath(), x.getHttpMethod().name()))
        .distinct()
        .collect(toList());

    List<ValidationResult> validationResults = new LinkedList<>();

    for (Pair<String, String> descriptorOperation : descriptorOperations) {
      if (apiOperations.stream().noneMatch(
                                           x -> x.getKey().equalsIgnoreCase(descriptorOperation.getKey())
                                               && x.getValue().equalsIgnoreCase(descriptorOperation.getValue()))) {
        validationResults.add(
                              new ValidationResult(this,
                                                   "Path: " + descriptorOperation.getKey() + " - Method: "
                                                       + descriptorOperation.getValue()));
      }
    }

    return validationResults;
  }
}
