/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.validation.rules;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.connectivity.restconnect.internal.validation.ValidationRule.Level.ERROR;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorModel;
import org.mule.connectivity.restconnect.internal.connectormodel.type.EmptyTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.schema.TypeSchema;
import org.mule.connectivity.restconnect.internal.descriptor.model.ConnectorDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.OperationDescriptor;
import org.mule.connectivity.restconnect.internal.validation.PostValidationRule;
import org.mule.connectivity.restconnect.internal.validation.ValidationResult;
import org.mule.connectivity.restconnect.internal.webapi.model.APIModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APIOperationModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APITypeModel;

import java.util.List;

import javax.ws.rs.core.MediaType;

public class PagedOperationsOperationMustDeclareResponseBodyRule extends PostValidationRule {

  public PagedOperationsOperationMustDeclareResponseBodyRule() {
    super("Paged Operations must declare a response body",
          "All paginated operations must declare a an output type.",
          ERROR);
  }

  @Override
  public List<ValidationResult> postValidate(ConnectorDescriptor connectorDescriptor,
                                             APIModel apiModel,
                                             ConnectorModel connectorModel) {

    List<APIOperationModel> nullOutputAPIOperations = apiModel.getOperationsModel().stream()
        .filter(this::hasNoTypeDefinition)
        .collect(toList());

    return nullOutputAPIOperations.stream()
        .filter(nullOutputOperation -> operationIsPaginated(connectorDescriptor, nullOutputOperation))
        .filter(nullOutputOperation -> !operationIsIgnored(connectorDescriptor, nullOutputOperation))
        .filter(nullOutputOperation -> !typeDefinedInDescriptor(connectorDescriptor, nullOutputOperation))
        .map(this::getValidationError)
        .collect(toList());
  }

  private boolean operationIsIgnored(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel) {
    OperationDescriptor operationDescriptor = connectorDescriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(apiOperationModel.getPath()))
        .flatMap(x -> x.getOperations().stream())
        .filter(x -> x.getMethod().equalsIgnoreCase(apiOperationModel.getHttpMethod().name()))
        .findFirst().orElse(null);

    return operationDescriptor != null && operationDescriptor.isIgnored();
  }

  private boolean hasNoTypeDefinition(APIOperationModel apiOperationModel) {
    if (apiOperationModel.getOutputMetadataModel().isEmpty()) {
      return true;
    }

    return apiOperationModel.getOutputMetadataModel().stream()
        .anyMatch(PagedOperationsOperationMustDeclareResponseBodyRule::hasNoTypeDefinition);
  }

  private static boolean hasNoTypeDefinition(APITypeModel outputType) {
    MediaType outputMediaType = outputType.getMediaType();

    if (outputMediaType != null && outputMediaType.equals(MediaType.TEXT_PLAIN_TYPE)) {
      return false;
    }

    Class<? extends TypeDefinition> outputClass = outputType.getTypeDefinitionClass();
    TypeSchema typeSchema = outputType.getAPITypeSchemaModel().getTypeSchema();

    return typeSchema == null || (outputClass == null || outputClass == EmptyTypeDefinition.class);
  }

  private boolean typeDefinedInDescriptor(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel) {
    OperationDescriptor operationDescriptor = connectorDescriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(apiOperationModel.getPath()))
        .flatMap(x -> x.getOperations().stream())
        .filter(x -> x.getMethod().equalsIgnoreCase(apiOperationModel.getHttpMethod().name()))
        .findFirst().orElse(null);

    return operationDescriptor != null && isNotBlank(operationDescriptor.getOutputTypeSchema());
  }

  private boolean operationIsPaginated(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel) {
    OperationDescriptor operationDescriptor = connectorDescriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(apiOperationModel.getPath()))
        .flatMap(x -> x.getOperations().stream())
        .filter(x -> x.getMethod().equalsIgnoreCase(apiOperationModel.getHttpMethod().name()))
        .findFirst().orElse(null);

    return operationDescriptor != null && isNotBlank(operationDescriptor.getPagination());
  }

  private ValidationResult getValidationError(APIOperationModel apiOperationModel) {
    String location =
        "Operation with PATH: " + apiOperationModel.getPath() + " and METHOD: " + apiOperationModel.getHttpMethod().name();
    return new ValidationResult(this, location);
  }
}
