/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.api;

import static com.google.common.base.Charsets.UTF_8;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.io.FileUtils.writeStringToFile;
import static org.apache.commons.lang.StringUtils.EMPTY;
import static org.mule.connectivity.restconnect.internal.connectormodel.ConnectorPackage.buildBasePackage;
import static org.mule.connectivity.restconnect.internal.util.NamingUtil.isFriendlyName;
import static org.mule.connectivity.restconnect.internal.util.NamingUtil.makeNameFriendly;
import static org.mule.connectivity.restconnect.internal.webapi.util.XmlUtils.getXmlName;
import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorCategory;
import org.mule.connectivity.restconnect.internal.connectormodel.HTTPMethod;
import org.mule.connectivity.restconnect.internal.connectormodel.type.EmptyTypeDefinition;
import org.mule.connectivity.restconnect.internal.descriptor.model.APISpecDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.APIUrlDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.BaseUriDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.EndPointDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.MavenGavDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.OperationDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.PaginationDeclarationDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.ParameterDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.RequestDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.ConnectorDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.SecuritySchemeDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.writer.DescriptorWriter;
import org.mule.connectivity.restconnect.internal.webapi.model.APIModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APIOperationModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APIParameterModel;
import org.mule.connectivity.restconnect.internal.webapi.parser.amf.AMFAPIModel;
import org.mule.connectivity.restconnect.internal.webapi.parser.ramlparser.RPAPIModel;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.List;

import org.mule.connectivity.restconnect.internal.webapi.util.XmlUtils;

public class DescriptorScaffolder {

  private final static String DEFAULT_MVN_GROUP_ID = "com.mulesoft.connectors";
  private final static String DEFAULT_MVN_VERSION = "1.0.0";

  private DescriptorScaffolder() {}

  public static void scaffoldDescriptor(File spec, SpecFormat format, Parser parser, Path outputDir)
      throws ModelGenerationException, IOException {
    ConnectorDescriptor connectorDescriptor = scaffoldDescriptorModel(spec, format, parser);
    writeToYaml(connectorDescriptor, outputDir);
  }

  public static ConnectorDescriptor scaffoldDescriptorModel(File spec, SpecFormat format, Parser parser)
      throws ModelGenerationException {
    APIModel apiModel = parser.equals(Parser.AMF)
        ? new AMFAPIModel(spec, spec.getParent(), format)
        : new RPAPIModel(spec, spec.getParent(), format);

    return generateConnectorDescriptor(spec.toPath(), apiModel);
  }

  public static void writeToYaml(ConnectorDescriptor connectorDescriptor, Path outputDir)
      throws IOException {
    DescriptorWriter descriptorWriter = new DescriptorWriter();
    String yaml = descriptorWriter.write(connectorDescriptor);

    writeStringToFile(outputDir.resolve("output.yaml").toFile(), yaml, UTF_8);
  }

  private static ConnectorDescriptor generateConnectorDescriptor(Path specPath, APIModel apiModel) {
    return new ConnectorDescriptor(
                                   generateAPISpec(specPath),
                                   generateApiName(apiModel),
                                   apiModel.getDescription(),
                                   generateConnectorGav(apiModel),
                                   generateBaseUri(apiModel),
                                   generateEndpoints(apiModel),
                                   generatePaginations(),
                                   generateSecurity(),
                                   ConnectorCategory.SELECT.toString(),
                                   generateDefaultPackage(apiModel),
                                   XmlUtils.getXmlName(generateApiName(apiModel)),
                                   null);
  }

  private static String generateDefaultPackage(APIModel apiModel) {
    return buildBasePackage(generateApiName(apiModel));
  }

  private static String generateApiName(APIModel apiModel) {
    return apiModel.getApiName().replaceAll("[\\s]+(?i)api[\\s]+", EMPTY)
        .replaceAll("^(?i)api[\\s]+", EMPTY)
        .replaceAll("[\\s]+(?i)api$", EMPTY);
  }

  private static List<SecuritySchemeDescriptor> generateSecurity() {
    return emptyList();
  }

  private static List<EndPointDescriptor> generateEndpoints(APIModel apiModel) {
    return apiModel.getOperationsModel().stream()
        .map(APIOperationModel::getPath)
        .distinct()
        .map(x -> generateEndpoint(x, apiModel))
        .collect(toList());
  }

  private static EndPointDescriptor generateEndpoint(String path, APIModel apiModel) {
    return new EndPointDescriptor(path, generateOperations(path, apiModel), false, null);
  }

  private static List<OperationDescriptor> generateOperations(String path, APIModel apiModel) {
    return apiModel.getOperationsModel().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(path))
        .map(DescriptorScaffolder::generateOperation)
        .collect(toList());
  }

  private static OperationDescriptor generateOperation(APIOperationModel operationModel) {
    return new OperationDescriptor(
                                   operationModel.getHttpMethod().name().toLowerCase(),
                                   generateOperationDisplayName(operationModel),
                                   operationModel.getDescription(),
                                   generateDefaultInputMediaType(operationModel),
                                   generateDefaultOutputMediaType(operationModel),
                                   generateRequest(operationModel),
                                   false,
                                   null,
                                   null,
                                   generateSkipOutputTypeValidation(operationModel),
                                   null,
                                   null);
  }

  private static String generateOperationDisplayName(APIOperationModel operationModel) {
    return isFriendlyName(operationModel.getName())
        ? operationModel.getName()
        : makeNameFriendly(operationModel.getName());
  }

  private static Boolean generateSkipOutputTypeValidation(APIOperationModel operationModel) {
    boolean methodRequiresBody =
        operationModel.getHttpMethod().equals(HTTPMethod.GET)
            || operationModel.getHttpMethod().equals(HTTPMethod.POST)
            || operationModel.getHttpMethod().equals(HTTPMethod.PATCH)
            || operationModel.getHttpMethod().equals(HTTPMethod.OPTIONS);

    if (methodRequiresBody && operationModel.getOutputMetadataModel().isEmpty())
      return false;

    boolean emptyBody = operationModel.getOutputMetadataModel().stream()
        .allMatch(x -> x.getTypeDefinitionClass() == null || x.getTypeDefinitionClass().equals(EmptyTypeDefinition.class));

    if (methodRequiresBody && emptyBody) {
      return false;
    }

    return null;
  }

  private static String generateDefaultInputMediaType(APIOperationModel operationModel) {
    if (!operationModel.getInputMetadataModel().isEmpty()) {
      return operationModel.getInputMetadataModel().get(0).getMediaType().toString();
    }
    return null;
  }

  private static String generateDefaultOutputMediaType(APIOperationModel operationModel) {
    if (!operationModel.getOutputMetadataModel().isEmpty()) {
      return operationModel.getOutputMetadataModel().get(0).getMediaType().toString();
    }
    return null;
  }

  private static RequestDescriptor generateRequest(APIOperationModel operationModel) {
    return new RequestDescriptor(generateParameters(operationModel.getHeadersModel()),
                                 generateParameters(operationModel.getQueryParamsModel()),
                                 generateParameters(operationModel.getUriParamsModel()));
  }

  private static List<ParameterDescriptor> generateParameters(List<APIParameterModel> parametersModel) {
    return parametersModel.stream().map(DescriptorScaffolder::generateParameter).collect(toList());
  }

  private static ParameterDescriptor generateParameter(APIParameterModel parameterModel) {
    return new ParameterDescriptor(generateParameterDisplayName(parameterModel),
                                   parameterModel.getExternalName(),
                                   parameterModel.getDescription());
  }

  private static String generateParameterDisplayName(APIParameterModel parameterModel) {
    return isFriendlyName(parameterModel.getDisplayName())
        ? parameterModel.getDisplayName()
        : makeNameFriendly(parameterModel.getDisplayName());
  }

  private static List<PaginationDeclarationDescriptor> generatePaginations() {
    return emptyList();
  }

  private static BaseUriDescriptor generateBaseUri(APIModel apiModel) {
    return new BaseUriDescriptor(apiModel.getBaseUri().getUri(), apiModel.getBaseUri().getType().getName(), emptyList());
  }

  private static MavenGavDescriptor generateConnectorGav(APIModel apiModel) {
    return new MavenGavDescriptor(DEFAULT_MVN_GROUP_ID,
                                  getXmlName(apiModel.getApiName()),
                                  DEFAULT_MVN_VERSION);
  }

  private static APISpecDescriptor generateAPISpec(Path specPath) {
    return new APIUrlDescriptor(specPath.toString());
  }
}
