/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.api;

import static java.lang.System.lineSeparator;
import static java.util.stream.Collectors.toList;
import static org.mule.connectivity.restconnect.internal.templating.TemplateEntity.getTemplateEngine;
import static org.mule.connectivity.restconnect.internal.validation.ModelValidationRules.Level.ERROR;
import static org.mule.connectivity.restconnect.internal.validation.ModelValidationRules.Level.INFO;
import static org.mule.connectivity.restconnect.internal.validation.ModelValidationRules.Level.WARN;
import static org.mule.connectivity.restconnect.internal.webapi.parser.ApiParser.parseModel;
import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.exception.TemplatingException;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorModel;
import org.mule.connectivity.restconnect.internal.connectormodel.builder.ConnectorModelBuilder;
import org.mule.connectivity.restconnect.internal.descriptor.model.ConnectorDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.parser.DescriptorParser;
import org.mule.connectivity.restconnect.internal.util.AMFWrapper;
import org.mule.connectivity.restconnect.internal.validation.ModelValidator;
import org.mule.connectivity.restconnect.internal.validation.ValidationResult;
import org.mule.connectivity.restconnect.internal.webapi.model.APIModel;

import java.io.File;
import java.nio.file.Path;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RestConnect {

  private static Logger logger = LoggerFactory.getLogger(RestConnect.class);

  private RestConnect() {}

  public static ConnectorModel getConnectorModel(File descriptor, File spec, SpecFormat format, Parser parser, String rootDir)
      throws ModelGenerationException {
    AMFWrapper.initialize();
    DescriptorParser descriptorParser = new DescriptorParser();
    ConnectorDescriptor connectorDescriptor = descriptorParser.parseConnectorDescriptor(descriptor);

    APIModel apiModel = parseModel(spec, format, parser, rootDir);

    ConnectorModelBuilder connectorModelBuilder = new ConnectorModelBuilder();
    ConnectorModel connectorModel = connectorModelBuilder.buildConnectorModel(apiModel, connectorDescriptor);

    if (validateModel(connectorDescriptor, apiModel, connectorModel)) {
      return connectorModel;
    }

    return null;
  }

  public static void generateConnector(ConnectorModel connectorModel, ConnectorType connectorType, Path outputDir,
                                       Path descriptorPath)
      throws TemplatingException {
    getTemplateEngine(connectorType, connectorModel, outputDir, descriptorPath).applyTemplates();
  }

  private static boolean validateModel(ConnectorDescriptor connectorDescriptor, APIModel apiModel,
                                       ConnectorModel connectorModel) {
    ModelValidator modelValidator = new ModelValidator();
    List<ValidationResult> validationResults = modelValidator.validate(connectorDescriptor, apiModel, connectorModel);

    List<ValidationResult> errorList = validationResults.stream()
        .filter(x -> x.getLevel().equals(ERROR))
        .collect(toList());

    List<ValidationResult> warnList = validationResults.stream()
        .filter(x -> x.getLevel().equals(WARN))
        .collect(toList());

    List<ValidationResult> infoList = validationResults.stream()
        .filter(x -> x.getLevel().equals(INFO))
        .collect(toList());

    errorList.forEach(x -> logger.error(x.toString() + lineSeparator()));
    warnList.forEach(x -> logger.warn(x.toString() + lineSeparator()));
    infoList.forEach(x -> logger.warn(x.toString() + lineSeparator()));

    return errorList.isEmpty();
  }
}
