/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.connectormodel;

import org.mule.connectivity.restconnect.exception.DuplicatedParameterNameException;
import org.mule.connectivity.restconnect.exception.InvalidSourceException;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.PartParameter;
import org.mule.connectivity.restconnect.internal.connectormodel.security.APISecurityScheme;
import org.mule.connectivity.restconnect.internal.connectormodel.type.MultipartTypeDefinition;
import org.mule.connectivity.restconnect.internal.connectormodel.type.TypeDefinition;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.connectivity.restconnect.internal.webapi.util.OperationNamingUtils.buildCanonicalOperationName;
import static org.mule.connectivity.restconnect.internal.webapi.util.XmlUtils.getXmlName;


public class ConnectorOperation {

  /**
   * XML name used to identify this operation internally.
   */
  private String internalName;

  /**
   * User friendly name for this parameter
   */
  private String displayName;

  private final String description;
  /**
   * Mapping key name defined for a multiple base path type.
   */
  private final String alternativeBaseUri;
  private final String path;
  private final HTTPMethod httpMethod;
  private final String pagination;
  private final List<Parameter> uriParameters;
  private final List<Parameter> queryParameters;
  private final List<Parameter> headers;
  private final List<Parameter> allParameters;

  private final TypeDefinition inputMetadata;
  private final TypeDefinition outputMetadata;

  private final Boolean skipOutputTypeValidation;

  private List<APISecurityScheme> securitySchemes;

  public ConnectorOperation(String displayName,
                            String description,
                            String path,
                            HTTPMethod httpMethod,
                            List<Parameter> uriParameters,
                            List<Parameter> queryParameters,
                            List<Parameter> headers,
                            TypeDefinition inputMetadata,
                            TypeDefinition outputMetadata,
                            List<APISecurityScheme> securitySchemes,
                            String alternativeBaseUri,
                            String pagination,
                            Boolean skipOutputTypeValidation)
      throws InvalidSourceException {

    this.internalName = getXmlName(buildCanonicalOperationName(httpMethod, path));
    this.displayName = displayName;
    this.description = description;
    this.path = path;
    this.httpMethod = httpMethod;

    this.uriParameters = uriParameters;
    this.queryParameters = queryParameters;
    this.headers = headers;
    this.inputMetadata = inputMetadata;
    this.outputMetadata = outputMetadata;
    this.securitySchemes = securitySchemes;

    this.allParameters = validateParameters();

    this.alternativeBaseUri = alternativeBaseUri;
    this.pagination = pagination;

    this.skipOutputTypeValidation = skipOutputTypeValidation;
  }

  private List<Parameter> validateParameters() throws InvalidSourceException {
    List<Parameter> allParams = new LinkedList<>();
    allParams.addAll(this.getQueryParameters());
    allParams.addAll(this.getUriParameters());
    allParams.addAll(this.getHeaders());
    allParams.addAll(this.getMultipartyBodyPartParameters());

    List<String> keys = new ArrayList<>();
    for (Parameter param : allParams) {
      if (keys.stream().anyMatch(str -> str.equals(param.getInternalName()))) {
        throw new InvalidSourceException("Invalid spec: Duplicated parameter names",
                                         new DuplicatedParameterNameException(param.getInternalName(), getInternalName(),
                                                                              getPath()));
      } else {
        keys.add(param.getInternalName());
      }
    }

    return allParams;
  }

  public List<Parameter> getQueryParameters() {
    return queryParameters;
  }

  public List<Parameter> getUriParameters() {
    return uriParameters;
  }

  public List<Parameter> getHeaders() {
    return headers;
  }

  private List<Parameter> getMultipartyBodyPartParameters() {
    List<Parameter> partParameters = new LinkedList<>();
    if (inputMetadata instanceof MultipartTypeDefinition) {
      for (PartParameter part : ((MultipartTypeDefinition) inputMetadata).getParts()) {
        if (part.getPartNameParameter() != null) {
          partParameters.add(part.getPartNameParameter());
        }
        if (part.getPartFilenameParameter() != null) {
          partParameters.add(part.getPartFilenameParameter());
        }
      }
    }
    return partParameters;
  }

  public String getHttpMethod() {
    return httpMethod.toString().toLowerCase();
  }

  public String getPath() {
    return path;
  }

  public List<Parameter> getAllParameters() {
    return allParameters;
  }

  public String getDescription() {
    return description;
  }

  public TypeDefinition getInputMetadata() {
    return inputMetadata;
  }

  public TypeDefinition getOutputMetadata() {
    return outputMetadata;
  }

  public List<APISecurityScheme> getSecuritySchemes() {
    return securitySchemes;
  }

  public String getInternalName() {
    return internalName;
  }

  public String getDisplayName() {
    return displayName;
  }

  public void setName(String displayName) {
    this.displayName = displayName;
    this.internalName = getXmlName(displayName);
  }

  public String getAlternativeBaseUri() {
    return alternativeBaseUri;
  }

  public String getPagination() {
    return pagination;
  }

  public boolean hasPagination() {
    return isNotBlank(this.pagination);
  }

  public Boolean getSkipOutputTypeValidation() {
    return skipOutputTypeValidation;
  }
}
