/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectivity.restconnect.internal.connectormodel.builder;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import org.mule.connectivity.restconnect.exception.ModelGenerationException;
import org.mule.connectivity.restconnect.internal.connectormodel.ConnectorOperation;
import org.mule.connectivity.restconnect.internal.connectormodel.parameter.Parameter;
import org.mule.connectivity.restconnect.internal.connectormodel.type.TypeDefinition;
import org.mule.connectivity.restconnect.internal.descriptor.model.ConnectorDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.EndPointDescriptor;
import org.mule.connectivity.restconnect.internal.descriptor.model.OperationDescriptor;
import org.mule.connectivity.restconnect.internal.webapi.model.APIModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APIOperationModel;
import org.mule.connectivity.restconnect.internal.webapi.model.APITypeModel;
import org.mule.connectivity.restconnect.internal.webapi.parser.TypeSchemaPool;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import javax.ws.rs.core.MediaType;

public class ConnectorOperationBuilder {

  private final ConnectorTypeDefinitionBuilder typeDefinitionBuilder;
  private final ConnectorParameterBuilder parameterBuilder;
  private final ConnectorSecuritySchemeBuilder securitySchemeBuilder;

  public ConnectorOperationBuilder(TypeSchemaPool typeSchemaPool) {
    this.typeDefinitionBuilder = new ConnectorTypeDefinitionBuilder(typeSchemaPool);
    this.parameterBuilder = new ConnectorParameterBuilder(typeDefinitionBuilder);
    this.securitySchemeBuilder = new ConnectorSecuritySchemeBuilder(typeDefinitionBuilder);
  }

  public List<ConnectorOperation> buildOperations(APIModel apiModel, ConnectorDescriptor connectorDescriptor)
      throws ModelGenerationException {

    List<ConnectorOperation> operations = new ArrayList<>();
    for (APIOperationModel op : apiModel.getOperationsModel()) {
      ConnectorOperation connectorOperation = buildOperation(op,
                                                             getEndpointDescriptor(op, connectorDescriptor),
                                                             getOperationDescriptor(op, connectorDescriptor));
      if (connectorOperation != null) {
        operations.add(connectorOperation);
      }
    }

    return operations;
  }

  private static EndPointDescriptor getEndpointDescriptor(APIOperationModel operationModel,
                                                          ConnectorDescriptor connectorDescriptor) {
    return connectorDescriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(operationModel.getPath()))
        .findFirst().orElse(null);
  }

  private static OperationDescriptor getOperationDescriptor(APIOperationModel operationModel,
                                                            ConnectorDescriptor connectorDescriptor) {
    return connectorDescriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(operationModel.getPath()))
        .flatMap(x -> x.getOperations().stream())
        .filter(x -> x.getMethod().equalsIgnoreCase(operationModel.getHttpMethod().name()))
        .findFirst().orElse(null);
  }

  public ConnectorOperation buildOperation(APIOperationModel apiOperationModel, EndPointDescriptor endPointDescriptor,
                                           OperationDescriptor operationDescriptor)
      throws ModelGenerationException {

    if (endPointDescriptor != null && endPointDescriptor.isIgnored()) {
      return null;
    }

    if (operationDescriptor != null && operationDescriptor.isIgnored()) {
      return null;
    }

    List<String> collisionNames = new LinkedList<>();
    List<Parameter> uriParameters =
        parameterBuilder.buildParameterList(apiOperationModel.getUriParamsModel(), operationDescriptor, collisionNames);
    collisionNames.addAll(uriParameters.stream().map(Parameter::getInternalName).collect(toList()));

    List<Parameter> queryParameters =
        parameterBuilder.buildParameterList(apiOperationModel.getQueryParamsModel(), operationDescriptor, collisionNames);
    collisionNames.addAll(queryParameters.stream().map(Parameter::getInternalName).collect(toList()));

    List<Parameter> headers =
        parameterBuilder.buildParameterList(apiOperationModel.getHeadersModel(), operationDescriptor, collisionNames);

    return new ConnectorOperation(
                                  buildOperationName(apiOperationModel, operationDescriptor),
                                  buildOperationDescription(apiOperationModel, operationDescriptor),
                                  apiOperationModel.getPath(),
                                  apiOperationModel.getHttpMethod(),
                                  uriParameters,
                                  queryParameters,
                                  headers,
                                  buildInputTypeMetadata(apiOperationModel, operationDescriptor),
                                  buildOutputTypeMetadata(apiOperationModel, operationDescriptor),
                                  securitySchemeBuilder.buildSecuritySchemes(apiOperationModel.getSecuritySchemesModel()),
                                  buildAlternativeBaseUri(endPointDescriptor, operationDescriptor),
                                  buildPagination(operationDescriptor),
                                  buildSkipOutputTypeValidation(operationDescriptor));
  }

  private static Boolean buildSkipOutputTypeValidation(OperationDescriptor operationDescriptor) {
    return operationDescriptor != null ? operationDescriptor.getSkipOutputTypeValidation() : null;
  }

  private static String buildOperationName(APIOperationModel operationModel, OperationDescriptor operationDescriptor) {
    if (operationDescriptor != null && isNotBlank(operationDescriptor.getName())) {
      return operationDescriptor.getName();
    } else {
      return operationModel.getName();
    }
  }

  private static String buildOperationDescription(APIOperationModel operationModel, OperationDescriptor operationDescriptor) {
    if (operationDescriptor != null && isNotBlank(operationDescriptor.getDescription())) {
      return operationDescriptor.getDescription();
    } else {
      return operationModel.getDescription();
    }
  }

  private static String buildAlternativeBaseUri(EndPointDescriptor endPointDescriptor, OperationDescriptor operationDescriptor) {
    if (operationDescriptor != null && isNotBlank(operationDescriptor.getBaseUri())) {
      return operationDescriptor.getBaseUri();
    } else if (endPointDescriptor != null && isNotBlank(endPointDescriptor.getBaseUri())) {
      return endPointDescriptor.getBaseUri();
    }
    return null;
  }

  private static String buildPagination(OperationDescriptor operationDescriptor) {
    if (operationDescriptor != null && isNotBlank(operationDescriptor.getPagination())) {
      return operationDescriptor.getPagination();
    }

    return null;
  }

  private TypeDefinition buildInputTypeMetadata(APIOperationModel apiOperationModel,
                                                OperationDescriptor operationDescriptor)
      throws ModelGenerationException {
    MediaType inputMediaType = null;

    if (operationDescriptor != null) {
      String inputMediaTypeString = operationDescriptor.getInputMediaType();
      if (isNotBlank(inputMediaTypeString)) {
        inputMediaType = MediaType.valueOf(inputMediaTypeString);
      }

      if (isNotBlank(operationDescriptor.getInputTypeSchema())) {
        return typeDefinitionBuilder.buildTypeDefinition(operationDescriptor.getInputTypeSchema(), inputMediaType);
      }
    }

    return buildTypeMetadata(apiOperationModel.getInputMetadataModel(), inputMediaType);
  }

  private TypeDefinition buildOutputTypeMetadata(APIOperationModel apiOperationModel,
                                                 OperationDescriptor operationDescriptor)
      throws ModelGenerationException {

    MediaType outputMediaType = null;

    if (operationDescriptor != null) {
      String outputMediaTypeString = operationDescriptor.getOutputMediaType();
      if (isNotBlank(outputMediaTypeString)) {
        outputMediaType = MediaType.valueOf(outputMediaTypeString);
      }

      if (isNotBlank(operationDescriptor.getOutputTypeSchema())) {
        return typeDefinitionBuilder.buildTypeDefinition(operationDescriptor.getOutputTypeSchema(), outputMediaType);
      }
    }

    return buildTypeMetadata(apiOperationModel.getOutputMetadataModel(), outputMediaType);
  }

  private TypeDefinition buildTypeMetadata(List<APITypeModel> metadataModels, MediaType mediaType)
      throws ModelGenerationException {

    if (!metadataModels.isEmpty()) {
      APITypeModel apiTypeModel = null;

      if (mediaType != null) {
        apiTypeModel = getApiTypeModelByMediaType(metadataModels, mediaType);
      }

      if (apiTypeModel == null) {
        apiTypeModel = getDefaultApiTypeModel(metadataModels);
      }

      return typeDefinitionBuilder.buildTypeDefinition(apiTypeModel);
    }

    return null;
  }

  private static APITypeModel getDefaultApiTypeModel(List<APITypeModel> metadataModels) {
    APITypeModel apiTypeModel = getApiTypeModelByMediaType(metadataModels, MediaType.APPLICATION_JSON_TYPE);

    if (apiTypeModel == null) {
      apiTypeModel = getApiTypeModelByMediaType(metadataModels, MediaType.APPLICATION_XML_TYPE);
    }

    return apiTypeModel == null ? metadataModels.get(0) : apiTypeModel;
  }

  private static APITypeModel getApiTypeModelByMediaType(List<APITypeModel> metadataModels, MediaType mediaType) {
    return metadataModels.stream().filter(x -> x.getMediaType().equals(mediaType)).findFirst().orElse(null);
  }
}
